/* This file is part of the KDE project
   Copyright (C) 2001,2002,2003,2004 Laurent Montel <montel@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/


#include "kchartFontConfigPage.h"

#include "kchartFontConfigPage.moc"

#include <tdeapplication.h>
#include <tdelocale.h>
#include <kcolorbutton.h>
#include <kdebug.h>

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqlineedit.h>
#include <tqlistbox.h>
#include <tqpushbutton.h>
#include <tqpainter.h>
#include <tqwhatsthis.h>

#include <tdefontdialog.h>

// For IRIX
namespace std {}

using namespace std;

#include "kchart_params.h"


class KChartFontListBoxItem : public TQListBoxText
{
public:
    KChartFontListBoxItem( TQListBox* lb,  const TQString& text = TQString() ) :
        TQListBoxText( lb,  text )  {}
    KChartFontListBoxItem( const TQString& text = TQString() ) :
        TQListBoxText( text )  {}

    void setFont( const TQFont& font )  {
        _font = font;
        listBox()->repaint();
    }
    TQFont font() const {
        return _font;
    }

protected:
    void paint( TQPainter* painter )  
    {
        painter->save();
        painter->setFont( _font );
        TQListBoxText::paint( painter );
        painter->restore();
    }

private:
    TQFont _font;
};


// ----------------------------------------------------------------


namespace KChart
{

KChartFontConfigPage::KChartFontConfigPage( KChartParams* params,
                                            TQWidget* parent, 
					    KDChartTableData *dat) :
    TQWidget( parent ), m_params( params ), data(dat)
{
    TQGridLayout *grid = new TQGridLayout(this,4,3,KDialog::marginHint(), KDialog::spacingHint());

    // The listbox
    m_list = new TQListBox(this);
    m_list->resize( m_list->sizeHint() );
    grid->addWidget(m_list, 0, 0); // Row 0-0, col 0-1

    // The font button.
    m_fontButton = new TQPushButton( this);
    m_fontButton->setText(i18n("Font..."));
    TQWhatsThis::add(m_fontButton, i18n("Select an item in the list above and click on this button to display the TDE font dialog in order to choose a new font for this item."));
    m_fontButton->resize( m_fontButton->sizeHint() );
    grid->addWidget( m_fontButton, 1, 0);

#if 0
    // FIXME: Integrate the font chooser instead?  Well, maybe later.
    TDEFontChooser  *fontChooser = new TDEFontChooser(this, "fontChooser");
    grid->addMultiCellWidget(fontChooser, 0, 2, 1, 1);
#endif

    grid->setColStretch(2, 1);
    grid->setRowStretch(3, 1);

    connect( m_fontButton, TQ_SIGNAL(clicked()),
	     this,         TQ_SLOT(changeLabelFont()));
    connect( m_list,       TQ_SIGNAL(doubleClicked ( TQListBoxItem * )),
	     this,         TQ_SLOT(changeLabelFont()));

    // Enter the items into the list.
    initList();
}


void KChartFontConfigPage::initList()
{
    if ( m_params->chartType() != KChartParams::Pie
	 && m_params->chartType() != KChartParams::Ring ) {
        m_list->insertItem(new KChartFontListBoxItem( i18n("X-Title")));
        m_list->insertItem(new KChartFontListBoxItem( i18n("Y-Title")));
        m_list->insertItem(new KChartFontListBoxItem( i18n("X-Axis")));
        m_list->insertItem(new KChartFontListBoxItem( i18n("Y-Axis")));
        m_list->insertItem(new KChartFontListBoxItem( i18n("All Axes")));
    }

    m_list->insertItem(i18n("Label"));
    m_list->setCurrentItem(0);
}



void KChartFontConfigPage::changeLabelFont()
{
    TQFont                 *font = 0;
    TQButton::ToggleState  *state = 0;
    bool                   diffAxes = false;

    if (m_list->currentText()==i18n("X-Title")) {
        font  = &xTitle;
        state = &xTitleIsRelative;
    } else if(m_list->currentText()==i18n("Y-Title")) {
        font  = &yTitle;
        state = &yTitleIsRelative;
    } else if(m_list->currentText()==i18n("X-Axis")) {
        font  = &xAxis;
        state = &xAxisIsRelative;
    } else if(m_list->currentText()==i18n("Y-Axis")) {
        font  = &yAxis;
        state = &yAxisIsRelative;
    } else if(m_list->currentText()==i18n("All Axes")) {
        diffAxes = true;
    } else if(m_list->currentText()==i18n("Label")) {
        font  = &label;
        state = &labelIsRelative;
    }
    else
        kdDebug( 35001 ) << "Pb in listBox" << endl;

    if ( diffAxes ) {
        TQFont newFont;
        int flags = 0;
        TQButton::ToggleState newState
            = (xAxisIsRelative == yAxisIsRelative)
            ? (xAxisIsRelative ? TQButton::On : TQButton::Off)
            : TQButton::NoChange;
        if (TDEFontDialog::getFontDiff( newFont,
                                      flags,
                                      false,
                                      this,
                                      true,
                                      &newState ) != TQDialog::Rejected) {
            if ( TDEFontChooser::FamilyList & flags ) {
                xAxis.setFamily( newFont.family() );
                yAxis.setFamily( newFont.family() );
            }

            if ( TDEFontChooser::StyleList & flags ) {
                xAxis.setWeight( newFont.weight() );
                xAxis.setItalic( newFont.italic() );
                xAxis.setUnderline( newFont.underline() );
                xAxis.setStrikeOut( newFont.strikeOut() );

                yAxis.setWeight( newFont.weight() );
                yAxis.setItalic( newFont.italic() );
                yAxis.setUnderline( newFont.underline() );
                yAxis.setStrikeOut( newFont.strikeOut() );
            }

            if ( TDEFontChooser::SizeList & flags ) {
                xAxis.setPointSize( newFont.pointSize() );
                yAxis.setPointSize( newFont.pointSize() );
            }

            // CharSet settings are ignored since we are not TQt 2.x compatible
            // if( TDEFontChooser::CharsetList & flags ) {
            // }

            if ( TQButton::NoChange != newState ) {
                xAxisIsRelative = newState;
                yAxisIsRelative = newState;
            }
        }
    }
    else if ( font && state ) {
        TQFont newFont( *font );
        TQButton::ToggleState newState = *state;
        if (TDEFontDialog::getFont( newFont,
                                  false,
                                  this,
                                  true,
                                  &newState ) != TQDialog::Rejected) {
            *font = newFont;
            if ( TQButton::NoChange != newState )
                *state = newState;
        }
    }
}


void KChartFontConfigPage::init()
{
    KDChartAxisParams  leftparms;
    leftparms   = m_params->axisParams( KDChartAxisParams::AxisPosLeft );
    KDChartAxisParams  rightparms;
    rightparms  = m_params->axisParams( KDChartAxisParams::AxisPosRight );
    KDChartAxisParams  bottomparms;
    bottomparms = m_params->axisParams( KDChartAxisParams::AxisPosBottom );

    xAxis           = bottomparms.axisLabelsFont();
    xAxisIsRelative = bottomparms.axisLabelsFontUseRelSize() 
	? TQButton::On : TQButton::Off;

    if ( TQButton::On == xAxisIsRelative )
        xAxis.setPointSize( bottomparms.axisLabelsFontRelSize() );

    yAxis = leftparms.axisLabelsFont();
    yAxisIsRelative = leftparms.axisLabelsFontUseRelSize()
	? TQButton::On : TQButton::Off;

    if ( TQButton::On == yAxisIsRelative )
        yAxis.setPointSize( leftparms.axisLabelsFontRelSize() );
    // PENDING(khz) Add support for the other 6 possible axes

    // PENDING(khz) Add support for the other 16 possible hd/ft areas


   xTitle = m_params->axisTitleFont( KDChartAxisParams::AxisPosBottom );
   yTitle = m_params->axisTitleFont( KDChartAxisParams::AxisPosLeft );
   xTitle.setPointSize( m_params->axisTitleFontRelSize( KDChartAxisParams::AxisPosBottom ) );
   yTitle.setPointSize( m_params->axisTitleFontRelSize( KDChartAxisParams::AxisPosLeft ) );
//   label = _params->labelFont();

    // PENDING(kalle) Adapt
//   for(int i=0;i<12;i++)
//     extColor.setColor(i,_params->ExtColor.color(i));
//   index = 0;
//   colorButton->setColor(extColor.color(index));
}


void KChartFontConfigPage::apply()
{
    // PENDING(kalle) Adapt
    KDChartAxisParams  leftparms;
    leftparms   = m_params->axisParams( KDChartAxisParams::AxisPosLeft );
    KDChartAxisParams  rightparms;
    rightparms  = m_params->axisParams( KDChartAxisParams::AxisPosRight );
    KDChartAxisParams  bottomparms;
    bottomparms = m_params->axisParams( KDChartAxisParams::AxisPosBottom );

    leftparms.setAxisLabelsFont( yAxis, TQButton::Off == yAxisIsRelative );
    if ( TQButton::On == yAxisIsRelative )
        leftparms.setAxisLabelsFontRelSize( yAxis.pointSize() );

    // PENDING(khz) change right axis handling
    // use left axis settings for the right axis as well
    //   (this must be changed, khz 14.12.2001)
    rightparms.setAxisLabelsFont( yAxis, TQButton::Off == yAxisIsRelative );
    if ( TQButton::On == yAxisIsRelative )
        rightparms.setAxisLabelsFontRelSize( yAxis.pointSize() );

    bottomparms.setAxisLabelsFont( xAxis, TQButton::Off == xAxisIsRelative );
    if ( TQButton::On == xAxisIsRelative )
        bottomparms.setAxisLabelsFontRelSize( xAxis.pointSize() );
    // PENDING(khz) Add support for the other 6 possible axes

    m_params->setAxisParams( KDChartAxisParams::AxisPosLeft, leftparms );
    m_params->setAxisParams( KDChartAxisParams::AxisPosRight, rightparms );
    m_params->setAxisParams( KDChartAxisParams::AxisPosBottom, bottomparms );
    // PENDING(khz) change hd2 and ft handling
    // use header settings for header 2 and footer as well
    //   (this must be changed, khz 14.12.2001)
    // PENDING(khz) Add support for the other 16 possible hd/ft areas

    m_params->setAxisTitleFont( KDChartAxisParams::AxisPosLeft, yTitle );
    m_params->setAxisTitleFont( KDChartAxisParams::AxisPosBottom, xTitle );
    m_params->setAxisTitleFontRelSize( KDChartAxisParams::AxisPosLeft, yTitle.pointSize() );
    m_params->setAxisTitleFontRelSize( KDChartAxisParams::AxisPosBottom, xTitle.pointSize() );
}

}  //KChart namespace
