/*
     rdphostpref.cpp, handles preferences for RDP hosts
     Copyright (C) 2003 Arend van Beelen jr.

     This program is free software; you can redistribute it and/or modify it under the terms of the
     GNU General Public License as published by the Free Software Foundation; either version 2 of
     the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
     without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
     the GNU General Public License for more details.

     You should have received a copy of the GNU General Public License along with this program; if
     not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
     MA 02110-1301 USA

     For any questions, comments or whatever, you may mail me at: arend@auton.nl
*/

#include "rdphostpref.h"
#include <tdeconfig.h>
#include <tdelocale.h>

const TQString RdpHostPref::RdpType = "RDP";

RdpHostPref::RdpHostPref(TDEConfig *conf, const TQString &host, const TQString &type) :
	HostPref(conf, host, type),
	m_width(0),
	m_height(0),
	m_colorDepth(24),
	m_layout("en-us"),
	m_askOnConnect(true),
	m_useTDEWallet(true)
{
}

RdpHostPref::~RdpHostPref()
{
}

void RdpHostPref::save()
{
	if ( !m_host.isEmpty() && !m_type.isEmpty() )
	{
		m_config->setGroup("PerHostSettings");
		TQString p = prefix();
		m_config->writeEntry(p+"exists", true);
		m_config->writeEntry(p+"width", m_width);
		m_config->writeEntry(p+"height", m_height);
		m_config->writeEntry(p+"colorDepth", m_colorDepth);
		m_config->writeEntry(p+"layout", m_layout);
		m_config->writeEntry(p+"askOnConnect", m_askOnConnect);
		m_config->writeEntry(p+"useTDEWallet", m_useTDEWallet);
	}
	else
	{
		m_config->setGroup( "RdpDefaultSettings" );
		m_config->writeEntry( "rdpWidth", m_width );
		m_config->writeEntry( "rdpHeight", m_height );
		m_config->writeEntry( "rdpColorDepth", m_colorDepth);
		m_config->writeEntry( "rdpKeyboardLayout", m_layout );
		m_config->writeEntry( "rdpShowHostPreferences", m_askOnConnect );
		m_config->writeEntry( "rdpUseTDEWallet", m_useTDEWallet );
	}
}

void RdpHostPref::load()
{
	if ( !m_host.isEmpty() && !m_type.isEmpty() )
	{
		m_config->setGroup("PerHostSettings");
		TQString p = prefix();
		m_width = m_config->readNumEntry(p+"width", 0);
		m_height = m_config->readNumEntry(p+"height", 0);
		m_colorDepth = m_config->readNumEntry(p+"colorDepth", 24);
		m_layout = m_config->readEntry(p+"layout", "en-us");
		m_askOnConnect = m_config->readBoolEntry(p+"askOnConnect", true);
		m_useTDEWallet = m_config->readBoolEntry(p+"useTDEWallet", true);
	}
	else
	{
		setDefaults();
	}
}

void RdpHostPref::remove()
{
	m_config->setGroup("PerHostSettings");
	TQString p = prefix();
	m_config->deleteEntry(p+"exists");
	m_config->deleteEntry(p+"width");
	m_config->deleteEntry(p+"height");
	m_config->deleteEntry(p+"colorDepth");
	m_config->deleteEntry(p+"layout");
	m_config->deleteEntry(p+"askOnConnect");
	m_config->deleteEntry(p+"useTDEWallet");
}

void RdpHostPref::setDefaults()
{
	m_config->setGroup("RdpDefaultSettings");
	m_width = m_config->readNumEntry("rdpWidth", 0);
	m_height = m_config->readNumEntry("rdpHeight", 0);
	m_colorDepth = m_config->readNumEntry("rdpColorDepth", 24);
	m_layout = m_config->readEntry("rdpKeyboardLayout", "en-us");
	m_askOnConnect = m_config->readBoolEntry("rdpShowHostPreferences", true);
	m_useTDEWallet = m_config->readBoolEntry("rdpUseTDEWallet", true);
}

TQString RdpHostPref::prefDescription() const
{
	return i18n("Show Preferences: %1, Resolution: %2x%3, Color Depth: %4, Keymap: %5, TDEWallet: %6")
	  .arg(m_askOnConnect ? i18n("yes") : i18n("no")).arg(m_width).arg(m_height)
	  .arg(m_colorDepth).arg(m_layout).arg(m_useTDEWallet ? i18n("yes") : i18n("no"));
}

void RdpHostPref::setWidth(int w)
{
	m_width = w;
	save();
}

int RdpHostPref::width() const
{
	return m_width;
}

void RdpHostPref::setHeight(int h)
{
	m_height = h;
	save();
}

int RdpHostPref::height() const
{
	return m_height;
}

void RdpHostPref::setColorDepth(int d)
{
	m_colorDepth = d;
	save();
}

int RdpHostPref::colorDepth() const
{
	return m_colorDepth;
}


void RdpHostPref::setLayout(const TQString &l)
{
	m_layout = l;
	save();
}

TQString RdpHostPref::layout() const
{
	return m_layout;
}

void RdpHostPref::setAskOnConnect(bool ask)
{
	m_askOnConnect = ask;
	save();
}

bool RdpHostPref::askOnConnect() const
{
	return m_askOnConnect;
}

void RdpHostPref::setUseTDEWallet(bool use) {
	m_useTDEWallet = use;
	save();
}

bool RdpHostPref::useTDEWallet() const {
	return m_useTDEWallet;
}
