/**********************************************************************
** Copyright (C) 2000-2001 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include <tqvariant.h>  // HP-UX compiler needs this here
#include <tqlistview.h>
#include "workspace.h"
#include "formwindow.h"
#include "mainwindow.h"
#include "pixmapchooser.h"
#include "globaldefs.h"
#include "command.h"
#ifndef KOMMANDER
#include "project.h"
#include "pixmapcollection.h"
#include "sourcefile.h"
#include "sourceeditor.h"
#endif
#include "formfile.h"

#include <tqheader.h>
#include <tqfileinfo.h>
#include <tqapplication.h>
#include <tqpainter.h>
#include <tqpen.h>
#include <tqobjectlist.h>
#include <tqworkspace.h>
#include <tqpopupmenu.h>
#include <tqtextstream.h>
#include "qcompletionedit.h"

#include <kurldrag.h>
#include <tdelocale.h>

static const char * const folderxpm[]={
    "16 16 6 1",
    ". c None",
    "b c #ffff00",
    "d c #000000",
    "* c #999999",
    "a c #cccccc",
    "c c #ffffff",
    "................",
    "................",
    "..*****.........",
    ".*ababa*........",
    "*abababa******..",
    "*cccccccccccc*d.",
    "*cbababababab*d.",
    "*cabababababa*d.",
    "*cbababababab*d.",
    "*cabababababa*d.",
    "*cbababababab*d.",
    "*cabababababa*d.",
    "*cbababababab*d.",
    "**************d.",
    ".dddddddddddddd.",
    "................"};

static const char * const file_xpm[]={
    "16 16 5 1",
    ". c #7f7f7f",
    "# c None",
    "c c #000000",
    "b c #bfbfbf",
    "a c #ffffff",
    "################",
    "..........######",
    ".aaaaaaaab.#####",
    ".aaaaaaaaba.####",
    ".aaaaaaaacccc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".aaaaaaaaaabc###",
    ".bbbbbbbbbbbc###",
    "ccccccccccccc###"};

static TQPixmap *folderPixmap = 0;
static TQPixmap *filePixmap = 0;
static TQPixmap* formPixmap = 0;

#ifndef KOMMANDER
WorkspaceItem::WorkspaceItem( TQListView *parent, Project* p )
#else
WorkspaceItem::WorkspaceItem( TQListView *parent )
#endif
    : TQListViewItem( parent )
{
    init();
#ifndef KOMMANDER
    project = p;
#endif
    t = ProjectType;
    setPixmap( 0, *folderPixmap );
    setExpandable( false );
}

#ifndef KOMMANDER
WorkspaceItem::WorkspaceItem( TQListViewItem *parent, SourceFile* sf )
    : TQListViewItem( parent )
{
    init();
    sourceFile = sf;
    t = SourceFileType;
    setPixmap( 0, *filePixmap );
}
#endif

WorkspaceItem::WorkspaceItem( TQListViewItem *parent, FormFile* ff, Type type )
    : TQListViewItem( parent )
{
    init();
    formFile = ff;
    t = type;
    if ( type ==  FormFileType ) {
  setPixmap( 0, *formPixmap );
  TQObject::connect( ff, TQ_SIGNAL( somethingChanged(FormFile*) ), listView(), TQ_SLOT( update(FormFile*) ) );
#ifndef KOMMANDER
  if ( formFile->supportsCodeFile() )
      (void) new WorkspaceItem( this, formFile, FormSourceType );
    } else if ( type == FormSourceType ) {
  setPixmap( 0, *filePixmap );
#endif
    }
}

WorkspaceItem::WorkspaceItem( TQListView *parent, FormFile* ff, Type type )
    : TQListViewItem( parent )
{
    init();
    formFile = ff;
    t = type;
    if ( type ==  FormFileType ) {
  setPixmap( 0, *formPixmap );
  TQObject::connect( ff, TQ_SIGNAL( somethingChanged(FormFile*) ), listView(), TQ_SLOT( update(FormFile*) ) );
#ifndef KOMMANDER
  if ( formFile->supportsCodeFile() )
      (void) new WorkspaceItem( this, formFile, FormSourceType );
    } else if ( type == FormSourceType ) {
  setPixmap( 0, *filePixmap );
#endif
    }
}


void WorkspaceItem::init()
{
    autoOpen = false;
    useOddColor = false;
#ifndef KOMMANDER
    project = 0;
    sourceFile = 0;
#endif
    formFile = 0;
}

void WorkspaceItem::paintCell( TQPainter *p, const TQColorGroup &cg, int column, int width, int align )
{
    TQColorGroup g( cg );
    g.setColor( TQColorGroup::Base, backgroundColor() );
    g.setColor( TQColorGroup::Foreground, TQt::black );

#ifndef KOMMANDER
    if ( type() == FormSourceType && !formFile->hasFormCode() ) {
  g.setColor( TQColorGroup::Text, listView()->palette().disabled().color( TQColorGroup::Text) );
  g.setColor( TQColorGroup::HighlightedText, listView()->palette().disabled().color( TQColorGroup::Text) );
    } else {
#endif
  g.setColor( TQColorGroup::Text, TQt::black );
#ifndef KOMMANDER
    }
#endif
    p->save();

    if ( isModified() ) {
  TQFont f = p->font();
  f.setBold( true );
  p->setFont( f );
    }

    TQListViewItem::paintCell( p, g, column, width, align );
    p->setPen( TQPen( cg.dark(), 1 ) );
    if ( column == 0 )
  p->drawLine( 0, 0, 0, height() - 1 );
    if ( listView()->firstChild() != this ) {
  if ( nextSibling() != itemBelow() && itemBelow()->depth() < depth() ) {
      int d = depth() - itemBelow()->depth();
      p->drawLine( -listView()->treeStepSize() * d, height() - 1, 0, height() - 1 );
  }
    }
    p->drawLine( 0, height() - 1, width, height() - 1 );
    p->drawLine( width - 1, 0, width - 1, height() );
    p->restore();
}

TQString WorkspaceItem::text( int column ) const
{
    if ( column != 0 )
  return TQListViewItem::text( column );
    switch( t ) {
#ifndef KOMMANDER
    case ProjectType:
  if ( project->isDummy() )
      return i18n("<No Project>" );
  return project->makeRelative( project->fileName() );
#endif
    case FormFileType:
  return formFile->formName() + ": " + formFile->fileName();
#ifndef KOMMANDER
    case FormSourceType:
  return formFile->codeFile();
    case SourceFileType:
  return sourceFile->fileName();
#endif
    }

    return TQString(); // shut up compiler
}

void WorkspaceItem::fillCompletionList( TQStringList& completion )
{
    switch( t ) {
    case ProjectType:
  break;
    case FormFileType:
  completion += formFile->formName();
  completion += formFile->fileName();
  break;
#ifndef KOMMANDER
    case FormSourceType:
  completion += formFile->codeFile();
  break;
    case SourceFileType:
  completion += sourceFile->fileName();
  break;
#endif
    }
}

bool WorkspaceItem::checkCompletion( const TQString& completion )
{
    switch( t ) {
    case ProjectType:
  break;
    case FormFileType:
  return  completion == formFile->formName()
     || completion == formFile->fileName();
#ifndef KOMMANDER
    case FormSourceType:
  return completion == formFile->codeFile();
    case SourceFileType:
  return completion == sourceFile->fileName();
#endif
    }
    return false;
}


bool WorkspaceItem::isModified() const
{
#ifndef KOMMANDER
    switch( t ) {
    case ProjectType:
      return project->isModified();
    case FormFileType:
  return formFile->isModified( FormFile::WFormWindow );
    case FormSourceType:
  return formFile->isModified( FormFile::WFormCode );
    case SourceFileType:
  return sourceFile->isModified();
    }
    return false; // shut up compiler
#else
    return formFile->isModified();
#endif
}

TQString WorkspaceItem::key( int column, bool ) const
{
    TQString key = text( column );
    if ( t == FormFileType )
  key.prepend( "0" );
    else
  key.prepend( "a" );
    return key;
}

TQColor WorkspaceItem::backgroundColor()
{
    bool b = useOddColor;
    if ( t == FormSourceType && parent() )
  b = ( ( WorkspaceItem*)parent() )->useOddColor;
    return b ? *backColor2 : *backColor1;
}


void WorkspaceItem::setOpen( bool b )
{
    TQListViewItem::setOpen( b );
    autoOpen = false;
}

void WorkspaceItem::setAutoOpen( bool b )
{
    TQListViewItem::setOpen( b );
    autoOpen = b;
}

Workspace::Workspace( TQWidget *parent, MainWindow *mw )
    : TQListView( parent, 0, WStyle_Customize | WStyle_NormalBorder | WStyle_Title |
     WStyle_Tool | WStyle_MinMax | WStyle_SysMenu ), mainWindow( mw ),
#ifndef KOMMANDER
  project( 0 ), completionDirty( false )
#else
  completionDirty( false )
#endif
{
    init_colors();

    setDefaultRenameAction( Accept );
    blockNewForms = false;
    bufferEdit = 0;
    header()->setStretchEnabled( true );
    header()->hide();
    setSorting( 0 );
    setResizePolicy( TQScrollView::Manual );
#ifndef KOMMANDER
    setIcon( PixmapChooser::loadPixmap( "logo" ) );
#endif
    TQPalette p( palette() );
    p.setColor( TQColorGroup::Base, TQColor( *backColor2 ) );
    (void)*selectedBack; // hack
    setPalette( p );
    addColumn( i18n("Files" ) );
    setAllColumnsShowFocus( true );
    connect( this, TQ_SIGNAL( mouseButtonClicked( int, TQListViewItem *, const TQPoint &, int ) ),
       this, TQ_SLOT( itemClicked( int, TQListViewItem *, const TQPoint& ) ) ),
    connect( this, TQ_SIGNAL( doubleClicked( TQListViewItem * ) ),
       this, TQ_SLOT( itemDoubleClicked( TQListViewItem * ) ) ),
    connect( this, TQ_SIGNAL( contextMenuRequested( TQListViewItem *, const TQPoint &, int ) ),
       this, TQ_SLOT( rmbClicked( TQListViewItem *, const TQPoint& ) ) ),
    setHScrollBarMode( AlwaysOff );
    setVScrollBarMode( AlwaysOn );
    viewport()->setAcceptDrops( true );
    setAcceptDrops( true );
    setColumnWidthMode( 1, Manual );

    if ( !folderPixmap ) {
  folderPixmap = new TQPixmap( folderxpm );
  filePixmap = new TQPixmap( file_xpm );
  formPixmap = new TQPixmap( PixmapChooser::loadPixmap( "form.xpm", PixmapChooser::Mini ) );
    }

#ifdef KOMMANDER
  makeConnections(mw);
#endif
}


#ifndef KOMMANDER
void Workspace::projectDestroyed( TQObject* o )
{
    if ( o == project ) {
  project = 0;
  clear();
    }
}
#endif

#ifndef KOMMANDER
void Workspace::setCurrentProject( Project *pro )
{
    if ( project == pro )
  return;
    if ( project ) {
  disconnect( project, TQ_SIGNAL( sourceFileAdded(SourceFile*) ), this, TQ_SLOT( sourceFileAdded(SourceFile*) ) );
  disconnect( project, TQ_SIGNAL( sourceFileRemoved(SourceFile*) ), this, TQ_SLOT( sourceFileRemoved(SourceFile*) ) );
  disconnect( project, TQ_SIGNAL( formFileAdded(FormFile*) ), this, TQ_SLOT( formFileAdded(FormFile*) ) );
  disconnect( project, TQ_SIGNAL( formFileRemoved(FormFile*) ), this, TQ_SLOT( formFileRemoved(FormFile*) ) );
  disconnect( project, TQ_SIGNAL( projectModified() ), this, TQ_SLOT( update() ) );
    }
    project = pro;
    connect( project, TQ_SIGNAL( sourceFileAdded(SourceFile*) ), this, TQ_SLOT( sourceFileAdded(SourceFile*) ) );
    connect( project, TQ_SIGNAL( sourceFileRemoved(SourceFile*) ), this, TQ_SLOT( sourceFileRemoved(SourceFile*) ) );
    connect( project, TQ_SIGNAL( formFileAdded(FormFile*) ), this, TQ_SLOT( formFileAdded(FormFile*) ) );
    connect( project, TQ_SIGNAL( formFileRemoved(FormFile*) ), this, TQ_SLOT( formFileRemoved(FormFile*) ) );
    connect( project, TQ_SIGNAL( destroyed(TQObject*) ), this, TQ_SLOT( projectDestroyed(TQObject*) ) );
    connect( project, TQ_SIGNAL( projectModified() ), this, TQ_SLOT( update() ) );
    clear();

    if ( bufferEdit )
  bufferEdit->clear();

    projectItem = new WorkspaceItem( this, project );

    projectItem->setOpen( true );

    for ( TQPtrListIterator<SourceFile> sources = project->sourceFiles();
    sources.current(); ++sources ) {
  SourceFile* f = sources.current();
  (void) new WorkspaceItem( projectItem, f );
    }

    for ( TQPtrListIterator<FormFile> forms = project->formFiles();
    forms.current(); ++forms ) {
  FormFile* f = forms.current();
  (void) new WorkspaceItem( projectItem, f );
    }

    updateColors();
    completionDirty = true;
}
#endif

#ifdef KOMMANDER
void Workspace::makeConnections(MainWindow *)
{
    // FIXME
}
#endif

#ifndef KOMMANDER
void Workspace::sourceFileAdded( SourceFile* sf )
{
    (void) new WorkspaceItem( projectItem, sf );
    Q_UNUSED(sf);
    updateColors();
}
#endif

#ifndef KOMMANDER
void Workspace::sourceFileRemoved( SourceFile* sf )
{
    delete findItem( sf );
    updateColors();
}
#endif

void Workspace::formFileAdded( FormFile* ff )
{
#ifndef KOMMANDER
  (void) new WorkspaceItem( projectItem, ff );
#else
  (void) new WorkspaceItem( this, ff );
#endif
    updateColors();
}

void Workspace::formFileRemoved( FormFile* ff )
{
    delete findItem( ff );
    updateColors();
}


void Workspace::update()
{
    completionDirty = true;
    triggerUpdate();
}

void Workspace::update( FormFile* ff )
{
    TQListViewItem* i = findItem( ff );
    if ( i ) {
  i->repaint();
  if ( (i = i->firstChild()) )
      i->repaint();
    }
}


void Workspace::activeFormChanged( FormWindow *fw )
{
    WorkspaceItem *i = findItem( fw->formFile() );
    if ( i ) {
  setCurrentItem( i );
  setSelected( i, true );
  if ( !i->isOpen() )
      i->setAutoOpen( true );
    }

    closeAutoOpenItems();

}

#ifndef KOMMANDER
void Workspace::activeEditorChanged( SourceEditor *se )
{
    if ( !se->object() )
  return;

    if ( se->formWindow() ) {
  WorkspaceItem *i = findItem( se->formWindow()->formFile() );
  if ( i && i->firstChild() ) {
      if ( !i->isOpen() )
    i->setAutoOpen( true );
      setCurrentItem( i->firstChild() );
      setSelected( i->firstChild(), true );
  }
    } else {
  WorkspaceItem *i = findItem( se->sourceFile() );
  if ( i ) {
      setCurrentItem( i );
      setSelected( i, true );
  }
    }

    closeAutoOpenItems();
}
#endif

WorkspaceItem *Workspace::findItem( FormFile* ff)
{
    TQListViewItemIterator it( this );
    for ( ; it.current(); ++it ) {
  if ( ( (WorkspaceItem*)it.current() )->formFile == ff )
      return (WorkspaceItem*)it.current();
    }
    return 0;
}

#ifndef KOMMANDER
WorkspaceItem *Workspace::findItem( SourceFile *sf )
{
    TQListViewItemIterator it( this );
    for ( ; it.current(); ++it ) {
  if ( ( (WorkspaceItem*)it.current() )->sourceFile == sf )
      return (WorkspaceItem*)it.current();
    }
    return 0;
}
#endif

void Workspace::closeAutoOpenItems()
{
    TQListViewItemIterator it( this );
    for ( ; it.current(); ++it ) {
  WorkspaceItem* i = (WorkspaceItem*) it.current();
  WorkspaceItem* ip = (WorkspaceItem*) i->parent();
  if ( i->type() == WorkspaceItem::FormSourceType ) {
      if ( !i->isSelected() && !ip->isSelected()
     && ip->isAutoOpen() ) {
    ip->setAutoOpen( false );
      }
  }
    }
}


void Workspace::closeEvent( TQCloseEvent *e )
{
    e->accept();
}

void Workspace::itemDoubleClicked( TQListViewItem *i )
{
    if ( ( (WorkspaceItem*)i)->type()== WorkspaceItem::ProjectType )
  i->setOpen( true );
}

void Workspace::itemClicked( int button, TQListViewItem *i, const TQPoint& )
{
    if ( !i || button != TQt::LeftButton )
  return;

    closeAutoOpenItems();

    WorkspaceItem* wi = (WorkspaceItem*)i;
#ifndef KOMMANDER
    if ( wi->type() == WorkspaceItem::SourceFileType )
  mainWindow->editSource( wi->sourceFile );
    switch( wi->type() ) {
    case WorkspaceItem::ProjectType:
  break; // ### TODO
    case WorkspaceItem::FormFileType:
  wi->formFile->showFormWindow();
  break;
    case WorkspaceItem::FormSourceType:
  wi->formFile->showEditor();
  break;
    case WorkspaceItem::SourceFileType:
  mainWindow->editSource( wi->sourceFile );
  break;
    }
#else
    wi->formFile->showFormWindow();
#endif
}

void Workspace::contentsDropEvent( TQDropEvent *e )
{
  if (!KURLDrag::canDecode(e))
  {
    e->ignore();
    return;
  }
  
  KURL::List files;
  KURLDrag::decode(e, files);
  if (files.isEmpty())
    return;
  
  for (KURL::List::Iterator it = files.begin(); it != files.end(); ++it)
  {
    if (!(*it).isLocalFile())
      continue;
    TQString fn = (*it).path();
    mainWindow->fileOpen(fn);
  }
}

void Workspace::contentsDragEnterEvent( TQDragEnterEvent *e )
{
    if ( !KURLDrag::canDecode( e ) )
  e->ignore();
    else
  e->accept();
}

void Workspace::contentsDragMoveEvent( TQDragMoveEvent *e )
{
    if ( !KURLDrag::canDecode( e ) )
  e->ignore();
    else
  e->accept();
}

void Workspace::rmbClicked( TQListViewItem *i, const TQPoint& pos )
{
    if ( !i )
  return;
    WorkspaceItem* wi = (WorkspaceItem*)i;
    enum { OPEN_SOURCE, REMOVE_SOURCE, OPEN_FORM, REMOVE_FORM, OPEN_FORM_SOURCE };
    TQPopupMenu menu( this );
    switch ( wi->type() ) {
#ifndef KOMMANDER
    case WorkspaceItem::SourceFileType:
  menu.insertItem( i18n("&Open Source File..." ), OPEN_SOURCE );
  menu.insertSeparator();
  menu.insertItem( PixmapChooser::loadPixmap( "edit-cut" ),
       i18n("&Remove Source File From Project" ), REMOVE_SOURCE );
  break;
#endif
    case WorkspaceItem::FormFileType:
  menu.insertItem( i18n("&Open Form..." ), OPEN_FORM );
  menu.insertSeparator();
  menu.insertItem( PixmapChooser::loadPixmap( "edit-cut" ),
#ifndef KOMMANDER
       i18n("&Remove Form From Project" ), REMOVE_FORM );
#else
       i18n("&Remove Form" ), REMOVE_FORM );
#endif
  break;
    case WorkspaceItem::FormSourceType:
  menu.insertItem( i18n("&Open Form Source..." ), OPEN_FORM_SOURCE );
  menu.insertSeparator();
  menu.insertItem( PixmapChooser::loadPixmap( "edit-cut" ),
#ifndef KOMMANDER
       i18n("&Remove Form From Project" ), REMOVE_FORM );
#else
       i18n("&Remove Form" ), REMOVE_FORM );
#endif
  break;
#ifndef KOMMANDER
    case WorkspaceItem::ProjectType:
  MainWindow::self->popupProjectMenu( pos );
  return;
#endif
    default:
  return;
    }
    switch ( menu.exec( pos ) ) {
#ifndef KOMMANDER
    case REMOVE_SOURCE:
  project->removeSourceFile( wi->sourceFile );
  break;
#endif
    case OPEN_SOURCE:
  itemClicked( TQt::LeftButton, i, pos );
  break;
#ifndef KOMMANDER
    case REMOVE_FORM: // FIXME
  project->removeFormFile( wi->formFile );
  break;
#endif
    case OPEN_FORM:
  itemClicked( TQt::LeftButton, i, pos );
  break;
    case OPEN_FORM_SOURCE:
  itemClicked( TQt::LeftButton, i, pos );
  break;
    default:
  break;
    }
}

bool Workspace::eventFilter( TQObject *o, TQEvent * e )
{
    // Reggie, on what type of events do we have to execute updateBufferEdit()
    if ( o == bufferEdit && e->type() != TQEvent::ChildRemoved )
  updateBufferEdit();
    return TQListView::eventFilter( o, e );
}

void Workspace::setBufferEdit( QCompletionEdit *edit )
{
    bufferEdit = edit;
    connect( bufferEdit, TQ_SIGNAL( chosen( const TQString & ) ),
       this, TQ_SLOT( bufferChosen( const TQString & ) ) );
    bufferEdit->installEventFilter( this );
}

void Workspace::updateBufferEdit()
{
    if ( !bufferEdit || !completionDirty )
  return;
    completionDirty = false;
#ifndef KOMMANDER
    TQStringList completion = MainWindow::self->projectFileNames();
    while ( it.current() ) {
  ( (WorkspaceItem*)it.current())->fillCompletionList( completion );
  ++it;
    }
    completion.sort();
    bufferEdit->setCompletionList( completion );
#endif
}

void Workspace::bufferChosen( const TQString &buffer )
{
    if ( bufferEdit )
  bufferEdit->setText( "" );

#ifndef KOMMANDER
    if ( MainWindow::self->projectFileNames().contains( buffer ) ) {
  MainWindow::self->setCurrentProjectByFilename( buffer );
  return;
    }
#endif

    TQListViewItemIterator it( this );
    while ( it.current() ) {
  if ( ( (WorkspaceItem*)it.current())->checkCompletion( buffer ) ) {
      itemClicked( TQt::LeftButton, it.current(), TQPoint() );
      break;
  }
  ++it;
    }
}

void Workspace::updateColors()
{
    TQListViewItem* i = firstChild();
    if ( i )
  i = i->firstChild();
    bool b = true;
    while ( i ) {
  WorkspaceItem* wi = ( WorkspaceItem*) i;
  i = i->nextSibling();
  wi->useOddColor = b;
  b = !b;
    }
}
#include "workspace.moc"
