/***************************************************************************
    copyright            : (C) 2003-2006 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#ifndef IMPORTER_H
#define IMPORTER_H

class TQWidget;

#include "../datavectors.h"

#include <tdelocale.h>
#include <kurl.h>

#include <tqobject.h>
#include <tqstring.h>

namespace Tellico {
  namespace Import {
    enum Options {
      ImportProgress = 1 << 5   // show progress bar
    };

/**
 * The top-level abstract class for importing other document formats into Tellico.
 *
 * The Importer classes import a file, and return a pointer to a newly created
 * @ref Data::Collection. Any errors or warnings are added to a status message queue.
 * The calling function owns the collection pointer.
 *
 * @author Robby Stephenson
 */
class Importer : public TQObject {
TQ_OBJECT
  

public:
  Importer() : TQObject(), m_options(ImportProgress) {}
  /**
   * The constructor should immediately load the contents of the file to be imported.
   * Any warnings or errors should be added the the status message queue.
   *
   * @param url The URL of the file to import
   */
  Importer(const KURL& url) : TQObject(), m_options(ImportProgress), m_urls(url) {}
  Importer(const KURL::List& urls) : TQObject(), m_options(ImportProgress), m_urls(urls) {}
  Importer(const TQString& text) : TQObject(), m_options(ImportProgress), m_text(text) {}
  /**
   */
  virtual ~Importer() {}

  /**
   * Returns a pointer to a @ref Data::Collection containing the contents of the imported file.
   * This function should probably only be called once, but the subclasses may cache the
   * collection. The collection should not be created until this function is called.
   *
   * @return A pointer to a @ref Collection created on the stack, or 0 if none could be created.
   */
  virtual Data::CollPtr collection() = 0;
  /**
   * Returns a string containing all the messages added to the queue in the course of loading
   * and importing the file.
   *
   * @return The status message
   */
  const TQString& statusMessage() const { return m_statusMsg; }
  /**
   * Returns a widget with the setting specific to this importer, or 0 if no
   * options are needed.
   *
   * @return A pointer to the setting widget
   */
  virtual TQWidget* widget(TQWidget*, const char*) { return 0; }
  /**
   * Checks to see if the importer can return a collection of this type
   *
   * @param type The collection type to check
   * @return Whether the importer could return a collection of that type
   */
  virtual bool canImport(int) const { return true; }
  /**
   * Validate the import settings
   */
  virtual bool validImport() const { return true; }
  virtual void setText(const TQString& text) { m_text = text; }
  long options() const { return m_options; }
  void setOptions(long options) { m_options = options; }
  /**
   * Returns a string useful for the ProgressManager
   */
  TQString progressLabel() const {
    if(url().isEmpty()) return i18n("Loading data..."); else return i18n("Loading %1...").arg(url().fileName());
  }

public slots:
  /**
   * The import action was changed in the import dialog
   */
  virtual void slotActionChanged(int) {}

protected:
  /**
   * Return the URL of the imported file.
   *
   * @return the file URL
   */
  KURL url() const { return m_urls.isEmpty() ? KURL() : m_urls[0]; }
  KURL::List urls() const { return m_urls; }
  TQString text() const { return m_text; }
  /**
   * Adds a message to the status queue.
   *
   * @param msg A string containing a warning or error.
   */
  void setStatusMessage(const TQString& msg) { if(!msg.isEmpty()) m_statusMsg += msg + TQChar(' '); }

  static const uint s_stepSize;

private:
  long m_options;
  KURL::List m_urls;
  TQString m_text;
  TQString m_statusMsg;
};

  } // end namespace
} // end namespace

#endif
