/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Radek Novacek <rnovacek@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <polkit/polkit.h>

#include "polkit-tqt-details.h"

#include <tqshared.h>
#include <tqstring.h>
#include <tqstringlist.h>


namespace PolkitTQt
{

//--------------------------------------
// Details::Data
//--------------------------------------

class Details::Data : public TQShared
{
  public:
    Data() : details(nullptr)
    {
    }

    Data(const Data &other) : details(other.details)
    {
      if (details)
      {
        g_object_ref(details);
      }
    }

    Data(PolkitDetails *_details) : details(_details)
    {
      if (details)
      {
        g_object_ref(details);
      }
    }

    ~Data()
    {
      if (details)
      {
        g_object_unref(details);
      }
    }

    PolkitDetails *details;
};

//--------------------------------------
// Details
//--------------------------------------

Details::Details() : d(new Data)
{
  d->details = polkit_details_new();
}

Details::Details(PolkitDetails *pkDetails) : d(new Data(pkDetails))
{
}

Details::Details(const Details &other) : d(other.d)
{
  d->ref();
}

Details& Details::operator=(const Details &other)
{
  if (d != other.d)
  {
    if (d->deref())
    {
      delete d;
    }
    d = other.d;
    d->ref();
  }
  return *this;
}

Details::~Details()
{
  if (d->deref())
  {
    delete d;
  }
}

TQString Details::lookup(const TQString &key) const
{
  const gchar *result = polkit_details_lookup(d->details, key.utf8().data());
  if (result != nullptr)
  {
    return TQString::fromUtf8(result);
  }
  else
  {
    return TQString::null;
  }
}

void Details::insert(const TQString &key, const TQString &value)
{
  polkit_details_insert(d->details, key.utf8().data(), value.utf8().data());
}

TQStringList Details::keys() const
{
  gchar **result = polkit_details_get_keys(d->details);
  TQStringList list;
  int len = g_strv_length(result);
  for (int i = 0; i < len; i++)
  {
    list.append(TQString::fromUtf8(result[i]));
  }
  g_strfreev(result);
  return list;
}

}
