/***************************************************************************
 *   Copyright (C) 1999-2001 by Bernd Gehrmann                             *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MakeItem_h
#define MakeItem_h

#include <tqstring.h>

enum EOutputLevel
{
	// appropriate to the ID's in the button group of settingswidget.ui
	eVeryShort = 0
	,eShort
	,eFull
};

class MakeItem
{
public:
	enum Type { Normal, Error, Warning, Diagnostic };
	MakeItem();
	MakeItem( const TQString& text );
	virtual ~MakeItem();

  enum DisplayModes
  {
    DelayedDisplay = 0, // item can be displayed later
    ImmDisplay = 1,     // item has to be displayed ASAP
    Append = 2          // item's text can be appended (append has been overloaded)
  };
  virtual int displayMode() const { return ImmDisplay; }
	virtual bool append( const TQString& ) { return false; }
	virtual Type type() { return Diagnostic; }
	virtual bool visible( EOutputLevel level ) { return level > eVeryShort; }
	virtual TQString text( EOutputLevel );
	virtual TQString formattedText( EOutputLevel, bool bright_bg );
	TQString icon();
	TQString color( bool bright_bg );

	static TQString br();

	TQString m_text;
};

class CommandItem : public MakeItem
{
public:
	CommandItem(const TQString command)
		: MakeItem( command )
	{}

	Type type() { return Diagnostic; }
	virtual bool visible( EOutputLevel ) { return true; }
};

class ExitStatusItem : public MakeItem
{
public:
	ExitStatusItem( bool normalExit, int exitStatus );

	Type type() { return m_normalExit && m_exitStatus == 0 ? Diagnostic : Error; }
	virtual bool visible( EOutputLevel ) { return true; }
	TQString text( EOutputLevel level );

private:
	bool m_normalExit;
	int m_exitStatus;
};

class DirectoryItem : public MakeItem
{
public:
	DirectoryItem( const TQString& dir, const TQString& text )
		: MakeItem( text )
		, directory( dir )
	{}

	Type type() { return Diagnostic; }

	static void setShowDirectoryMessages( bool show ) { m_showDirectoryMessages = show; }
	static bool getShowDirectoryMessages() { return m_showDirectoryMessages; }

	TQString directory;

protected:
	static bool m_showDirectoryMessages;
};

class EnteringDirectoryItem : public DirectoryItem
{
public:
	EnteringDirectoryItem( const TQString& dir, const TQString& text )
		: DirectoryItem( dir, text )
	{}
	bool visible( EOutputLevel )
	{
		return m_showDirectoryMessages;
	}

	virtual TQString text( EOutputLevel );
};

class ExitingDirectoryItem : public DirectoryItem
{
public:
	ExitingDirectoryItem( const TQString& dir, const TQString& text )
		: DirectoryItem( dir, text )
	{}
	bool visible( EOutputLevel level )
	{
		return m_showDirectoryMessages && level > eVeryShort;
	}

	virtual TQString text( EOutputLevel );
};

namespace KTextEditor { class Cursor; class Document; }

class ErrorItem : public MakeItem
{
public:
	ErrorItem( const TQString& fn, int ln, const TQString& tx, const TQString& line, bool isWarning, bool isInstatiationInfo, const TQString& compiler );
	virtual ~ErrorItem();

	virtual bool append( const TQString& text );
  virtual int displayMode() const { return DelayedDisplay | Append; }
	Type type() { return m_isInstatiationInfo ? Diagnostic : (m_isWarning ? Warning : Error); }
	virtual bool visible( EOutputLevel ) { return true; }

	TQString fileName;
	int lineNum;
	TQString m_error;
	bool m_isWarning;
	bool m_isInstatiationInfo; ///this also implies isWarning
	TQString m_compiler;
};

class ActionItem : public MakeItem
{
public:
	ActionItem( const TQString& action, const TQString& file, const TQString& tool, const TQString& line )
		: MakeItem( line )
		, m_action( action )
		, m_file( file )
		, m_tool( tool )
	{}

	virtual bool visible( EOutputLevel ) { return true; }
	virtual TQString text( EOutputLevel level );

	TQString m_action;
	TQString m_file;
	TQString m_tool;
};

#endif
