/* This file is part of the KDE project
   Copyright (C) 2003 Julian Rockey <linux@jrockey.com>
   Copyright (C) 2003 Alexander Dymo <cloudtemple@mksat.net>
   Copyright (C) 2003 Mario Scalas <mario.scalas@libero.it>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "urlutil.h"

#include <tqstringlist.h>

#include <tqdir.h>
#include <tqfileinfo.h>
#include <kdebug.h>

#include <unistd.h>
#include <limits.h>
#include <stdlib.h>

#include <tdeversion.h>
#if (TDE_VERSION_MINOR==0) && (TDE_VERSION_MAJOR==3)
#include <kdevkurl.h>
#endif

///////////////////////////////////////////////////////////////////////////////
// Namespace URLUtil
///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::filename(const TQString & name) {
  int slashPos = name.findRev("/");
  return slashPos<0 ? name : name.mid(slashPos+1);
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::directory(const TQString & name) {
  int slashPos = name.findRev("/");
  return slashPos<0 ? TQString("") : name.left(slashPos);
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::relativePath(const KURL & parent, const KURL & child, uint slashPolicy) {
  bool slashPrefix = slashPolicy & SLASH_PREFIX;
  bool slashSuffix = slashPolicy & SLASH_SUFFIX;
  if (parent == child)
    return slashPrefix ? TQString("/") : TQString("");

  if (!parent.isParentOf(child)) return TQString();
  int a=slashPrefix ? -1 : 1;
  int b=slashSuffix ? 1 : -1;
  return child.path(b).mid(parent.path(a).length());
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::relativePath(const TQString & parent, const TQString & child, uint slashPolicy) {
  return relativePath(KURL(parent), KURL(child), slashPolicy);
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::upDir(const TQString & path, bool slashSuffix) {
  int slashPos = path.findRev("/");
  if (slashPos<1) return TQString();
  return path.mid(0,slashPos+ (slashSuffix ? 1 : 0) );
}

///////////////////////////////////////////////////////////////////////////////

KURL URLUtil::mergeURL(const KURL & source, const KURL & dest, const KURL & child) {

  // if already a child of source, then fine
  if (source.isParentOf(child) || source == child) return child;

  // if not a child of dest, return blank URL (error)
  if (!dest.isParentOf(child) && dest != child) return KURL();

  // if child is same as dest, return source
  if (dest == child) return source;

  // calculate
  TQString childUrlStr = child.url(-1);
  TQString destStemStr = dest.url(1);
  TQString sourceStemStr = source.url(1);
  return KURL(sourceStemStr.append( childUrlStr.mid( destStemStr.length() ) ) );

}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::getExtension(const TQString & path) {
  int dotPos = path.findRev('.');
  if (dotPos<0) return TQString("");
  return path.mid(dotPos+1);
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::extractPathNameRelative(const KURL &baseDirUrl, const KURL &url )
{
  TQString absBase = extractPathNameAbsolute( baseDirUrl ),
    absRef = extractPathNameAbsolute( url );
  int i = absRef.find( absBase, 0, true );

  if (i == -1)
    return TQString();

  if (absRef == absBase)
    return TQString( "." );
  else
    return absRef.replace( 0, absBase.length(), TQString() );
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::extractPathNameRelative(const TQString &basePath, const KURL &url )
{
#if (TDE_VERSION_MINOR!=0) || (TDE_VERSION_MAJOR!=3)
  KURL baseDirUrl = KURL::fromPathOrURL( basePath );
#else
  KURL baseDirUrl = KdevKURL::fromPathOrURL( basePath );
#endif
  return extractPathNameRelative( baseDirUrl, url );
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::extractPathNameRelative(const TQString &basePath, const TQString &absFilePath )
{
#if (TDE_VERSION_MINOR!=0) || (TDE_VERSION_MAJOR!=3)
  KURL baseDirUrl = KURL::fromPathOrURL( basePath ),
       fileUrl = KURL::fromPathOrURL( absFilePath );
#else
  KURL baseDirUrl = KdevKURL::fromPathOrURL( basePath ),
       fileUrl = KdevKURL::fromPathOrURL( absFilePath );
#endif
  return extractPathNameRelative( baseDirUrl, fileUrl );
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::extractPathNameAbsolute( const KURL &url )
{
  if (isDirectory( url ))
    return url.path( +1 ); // with trailing "/" if none is present
  else
  {
    // Ok, this is an over-tight pre-condition on "url" since I hope nobody will never
    // stress this function with absurd cases ... but who knows?
  /*
    TQString path = url.path();
    TQFileInfo fi( path );  // Argh: TQFileInfo is back ;))
    return ( fi.exists()? path : TQString() );
  */
    return url.path();
  }
}

///////////////////////////////////////////////////////////////////////////////

bool URLUtil::isDirectory( const KURL &url )
{
  return isDirectory( url.path() );
}

///////////////////////////////////////////////////////////////////////////////

bool URLUtil::isDirectory( const TQString &absFilePath )
{
  return TQDir( absFilePath ).exists();
}

///////////////////////////////////////////////////////////////////////////////

void URLUtil::dump( const KURL::List &urls, const TQString &aMessage )
{
  if (!aMessage.isNull())
  {
    kdDebug(9000) << aMessage << endl;
  }
  kdDebug(9000) << " List has " << urls.count() << " elements." << endl;

  for (size_t i = 0; i<urls.count(); ++i)
  {
    KURL url = urls[ i ];
//    kdDebug(9000) << " * Element = "  << url.path() << endl;
  }
}

///////////////////////////////////////////////////////////////////////////////

TQStringList URLUtil::toRelativePaths( const TQString &baseDir, const KURL::List &urls)
{
  TQStringList paths;

  for (size_t i=0; i<urls.count(); ++i)
  {
    paths << extractPathNameRelative( baseDir, urls[i] );
  }

  return paths;
}

///////////////////////////////////////////////////////////////////////////////

TQString URLUtil::relativePathToFile( const TQString & dirUrl, const TQString & fileUrl )
{
  if (dirUrl.isEmpty() || (dirUrl == "/"))
    return fileUrl;

  TQStringList dir = TQStringList::split("/", dirUrl, false);
  TQStringList file = TQStringList::split("/", fileUrl, false);

  TQString resFileName = file.last();
  file.remove(file.last());

  uint i = 0;
  while ( (i < dir.count()) && (i < (file.count())) && (dir[i] == file[i]) )
    i++;

  TQString result_up;
  TQString result_down;
  TQString currDir;
  TQString currFile;
  do
  {
    i >= dir.count() ? currDir = "" : currDir = dir[i];
    i >= file.count() ? currFile = "" : currFile = file[i];
    tqWarning("i = %d, currDir = %s, currFile = %s", i, currDir.latin1(), currFile.latin1());
    if (currDir.isEmpty() && currFile.isEmpty())
      break;
    else if (currDir.isEmpty())
      result_down += file[i] + '/';
    else if (currFile.isEmpty())
      result_up += "../";
    else
    {
      result_down += file[i] + '/';
      result_up += "../";
    }
    i++;
  }
  while ( (!currDir.isEmpty()) || (!currFile.isEmpty()) );

  return result_up + result_down + resFileName;
}

///////////////////////////////////////////////////////////////////////////////

// code from qt-3.1.2 version of TQDir::canonicalPath()
TQString URLUtil::canonicalPath( const TQString & path )
{
    TQString r;
    char cur[PATH_MAX+1];
    if ( ::getcwd( cur, PATH_MAX ) )
    {
        char tmp[PATH_MAX+1];
        if( ::realpath( TQFile::encodeName( path ), tmp ) )
        {
            r = TQFile::decodeName( tmp );
        }
        //always make sure we go back to the current dir
        ::chdir( cur );
    }
    return r;
}

///////////////////////////////////////////////////////////////////////////////

//written by "Dawit A." <adawit@kde.org>
//borrowed from his patch to KShell
TQString URLUtil::envExpand ( const TQString& str )
{
    uint len = str.length();

    if (len > 1 && str[0] == '$')
    {
      int pos = str.find ('/');

      if (pos < 0)
        pos = len;

      char* ret = getenv( TQConstString(str.unicode()+1, pos-1).string().local8Bit().data() );

      if (ret)
      {
        TQString expandedStr ( TQFile::decodeName( ret ) );
        if (pos < (int)len)
          expandedStr += str.mid(pos);
        return expandedStr;
      }
    }

    return str;
}

