#ifndef BUGCOMMAND_H
#define BUGCOMMAND_H

#include <tqstring.h>
#include <tqstringlist.h>

#include "bug.h"
#include "package.h"

class TDEConfig;

class BugCommand {
  public:
    enum Mode  { Normal, Maintonly, Quiet };
    enum State { None, Queued, Sent };

    BugCommand( const Bug &bug ) : m_bug( bug ) {}
    BugCommand( const Bug &bug, const Package &pkg ) : m_bug( bug ), m_package( pkg ) {}
    virtual ~BugCommand() {}

    virtual TQString controlString() const { return TQString(); }

    virtual TQString mailAddress() const { return TQString(); }
    virtual TQString mailText() const { return TQString(); }

    Bug bug() const { return m_bug; }
    Package package() const { return m_package; }

    virtual TQString name();
    virtual TQString details();

    virtual TQString type() const { return TQString(); }

    virtual void save( TDEConfig * ) = 0;
    static BugCommand *load( TDEConfig *, const TQString &type );

  protected:
    Bug m_bug;
    Package m_package;
};

class BugCommandClose : public BugCommand {
  public:
    BugCommandClose( const Bug &bug, const TQString &message, const Package &pkg )  :
        BugCommand( bug, pkg ), m_message( message ) {}

    TQString controlString() const;
    TQString mailAddress() const;
    TQString mailText() const;

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("Close"); }

    void save( TDEConfig * );

  private:
    TQString m_message;
};

class BugCommandCloseSilently : public BugCommand {
  public:
    BugCommandCloseSilently( const Bug &bug, const Package &pkg )  :
        BugCommand( bug, pkg ) {}

    TQString controlString() const;

    TQString name();

    TQString type() const { return TQString::fromLatin1("CloseSilently"); }

    void save( TDEConfig * );
};

class BugCommandReopen : public BugCommand {
  public:
    BugCommandReopen( const Bug &bug, const Package &pkg )  :
        BugCommand( bug, pkg ) {}

    TQString controlString() const;

    TQString name();

    TQString type() const { return TQString::fromLatin1("Reopen"); }

    void save( TDEConfig * );
};

class BugCommandRetitle : public BugCommand {
  public:
    BugCommandRetitle( const Bug &bug, const TQString &title, const Package &pkg )  :
        BugCommand( bug, pkg ), m_title( title ) {}

    TQString controlString() const;

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("Retitle"); }

    void save( TDEConfig * );

  private:
    TQString m_title;
};

class BugCommandMerge : public BugCommand {
  public:
    BugCommandMerge( const TQStringList &bugNumbers, const Package &pkg )  :
        BugCommand( Bug(), pkg ), m_bugNumbers( bugNumbers ) {}

    TQString controlString() const;

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("Merge"); }

    void save( TDEConfig * );

  private:
    TQStringList m_bugNumbers;
};

class BugCommandUnmerge : public BugCommand {
  public:
    BugCommandUnmerge( const Bug &bug, const Package &pkg )  :
        BugCommand( bug, pkg ) {}

    TQString name();

    TQString type() const { return TQString::fromLatin1("Unmerge"); }

    void save( TDEConfig * );

    TQString controlString() const;
};

class BugCommandReply : public BugCommand {
  public:
    BugCommandReply( const Bug &bug, const TQString &message, const int &recipient) :
        BugCommand( bug ), m_message( message ), m_recipient( recipient ) {}

    TQString mailAddress() const;
    TQString mailText() const;

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("Reply"); }

    void save( TDEConfig * );

  private:
    TQString m_message;
    int m_recipient;
};

class BugCommandReplyPrivate : public BugCommand {
  public:
    BugCommandReplyPrivate( const Bug &bug, const TQString &address,
                            const TQString &message ) :
         BugCommand( bug ), m_address( address ), m_message( message ) {}

    TQString mailAddress() const;
    TQString mailText() const;

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("ReplyPrivate"); }

    void save( TDEConfig * );

  private:
    TQString m_address;
    TQString m_message;
};

class BugCommandSeverity : public BugCommand {
  public:
    BugCommandSeverity( const Bug &bug, const TQString &severity, const Package &pkg ) :
        BugCommand( bug, pkg ), m_severity( severity ) {}

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("Severity"); }

    TQString controlString() const;

    void save( TDEConfig * );

  private:
    TQString m_severity;
};

class BugCommandReassign : public BugCommand {
  public:
    BugCommandReassign( const Bug &bug, const TQString &package, const Package &pkg ) :
        BugCommand( bug, pkg ), m_package( package ) {}

    TQString name();
    TQString details() const;

    TQString type() const { return TQString::fromLatin1("Reassign"); }

    TQString controlString() const;

    void save( TDEConfig * );

  private:
    TQString m_package;
};

#endif
