/*
    yahooaccount.h - Manages a single Yahoo account

    Copyright (c) 2003 by Gav Wood               <gav@kde.org>
    Copyright (c) 2003 by Matt Rogers            <mattrogers@sbcglobal.net>
    Based on code by Olivier Goffart             <ogoffart@kde.org>

    Kopete    (c) 2003 by the Kopete developers  <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/


#ifndef YAHOOACCOUNT_H
#define YAHOOACCOUNT_H

// TQt
#include <tqobject.h>
#include <tqmap.h>
#include <tqpixmap.h>
#include <tqpair.h>

// KDE
#include <kurl.h>

// Kopete
#include "kopetepasswordedaccount.h"

// Local
#include "yahooprotocol.h"
#include "yahootypes.h"

class TQColor;
class TDEAction;
class TDEActionMenu;
class YahooContact;
class YahooAccount;
class YahooProtocol;
class YahooWebcam;
class YahooConferenceChatSession;
class YahooChatChatSession;
class KTemporaryFile;

namespace Kopete{
class Transfer;
class ChatSession;
class StatusMessage;
class FileTransferInfo;
}
class Client;
struct YABEntry;
class KJob;
namespace TDEIO{
	class Job;
}

class YahooAccount : public Kopete::PasswordedAccount
{
	TQ_OBJECT
  

public:

	enum SignalConnectionType { MakeConnections, DeleteConnections };

	YahooAccount(YahooProtocol *parent,const TQString& accountID, const char *name = 0);
	~YahooAccount();

	/*
	 * Returns a contact of name @p id
	 */
	YahooContact *contact(const TQString &id);

	virtual TDEActionMenu* actionMenu();

	/**
	 * Sets the yahoo away status
	 */
	virtual void setAway(bool, const TQString &);

	/**
	 * The session
	 */
	Client *yahooSession();

	/**
	 * Returns true if contact @p id is on the server-side contact list
	 */
	bool isOnServer(const TQString &id) { return IDs.contains(id); }

	/**
	 * Returns true if we have the server-side contact list
	 */
	bool haveContactList() const { return theHaveContactList; }

	void setUseServerGroups(bool newSetting);

	void setImportContacts(bool newSetting);

	/**
	 * Set the pager server
	 */
	void setServer( const TQString &server );

	/**
	 * Set the port of the pager server
	 */
	void setPort( int port );

	/**
	 * Set Buddy Icon
	 */
	void setBuddyIcon( const KURL &url );

	void verifyAccount( const TQString &word );

	void sendConfMessage( YahooConferenceChatSession *s, const Kopete::Message &message );
	void sendChatMessage( const Kopete::Message &msg, const TQString &handle );
	void prepareConference( const TQString &who );
	void sendFile( YahooContact *to, const KURL &url );
public slots:
	/**
	 * Connect to the Yahoo service
	 */
	virtual void connectWithPassword( const TQString & );
	/**
	 * Disconnect from the Yahoo service
	 */
	virtual void disconnect();

	/** Reimplemented from Kopete::Account */
	void setOnlineStatus( const Kopete::OnlineStatus&, const TQString &reason = TQString());

signals:
	/**
	 * Emitted when we receive notification that the person we're talking to is typing
	 */
	void receivedTypingMsg(const TQString &contactId, bool isTyping);

	/**
	 * Emitted when our Buddy Icon has changed
	 */
	void signalBuddyIconChanged( int type );

protected:
	/**
	 * Adds our Yahoo contact to a metacontact
	 */
	virtual bool createContact(const TQString &contactId,  Kopete::MetaContact *parentContact);

	virtual bool createChatContact( const TQString &nick );

	/**
	 * Gets the just-received message color
	 */
	TQColor getMsgColor(const TQString& msg);
	/**
	 * Remove color codes from a message
	 */
	TQString stripMsgColorCodes(const TQString& msg);

protected slots:
	void slotConnected();
	void slotGoOnline();
	void slotGoOffline();
	void slotOpenInbox();			// Open Yahoo Mailbox in browser
	void slotOpenYAB();			// Open Yahoo Addressbook in browser
	void slotEditOwnYABEntry();		// Show own Yahoo Addressbook entry
	void slotJoinChatRoom();
	void slotChatCategorySelected( const Yahoo::ChatCategory &category );

	void slotGoStatus(int status, const TQString &awayMessage = TQString());
	void slotLoginResponse(int succ, const TQString &url);
	void slotDisconnected();
	void slotLoginFailed();
	void slotGotBuddy(const TQString &userid, const TQString &alias, const TQString &group);
	void slotBuddyAddResult(const TQString &userid, const TQString &group, bool success);
	void slotBuddyRemoveResult(const TQString &userid, const TQString &group, bool success);
	void slotBuddyChangeGroupResult(const TQString &userid, const TQString &group, bool success);
	void slotAuthorizationAccepted( const TQString &who );
	void slotAuthorizationRejected( const TQString &who, const TQString &msg );
	void slotgotAuthorizationRequest( const TQString &, const TQString &, const TQString & );
	void slotContactAddedNotifyDialogClosed( const TQString &user );
	//void slotAddedInfoEventActionActivated( uint actionId );
	void slotGotIgnore(const TQStringList &);
	void slotGotIdentities(const TQStringList &);
	void slotStatusChanged(const TQString &who, int stat, const TQString &msg, int away, int idle, int pictureChecksum);
	void slotStealthStatusChanged(const TQString &who, Yahoo::StealthStatus state);
	void slotGotIm(const TQString &who, const TQString &msg, long tm, int stat);
	void slotGotBuzz(const TQString &who, long tm);
	void slotGotConfInvite(const TQString &who, const TQString &room, const TQString &msg, const TQStringList &members);
	void slotConfUserDecline(const TQString &who, const TQString &room, const TQString &msg);
	void slotConfUserJoin(const TQString &who, const TQString &room);
	void slotConfUserLeave(const TQString &who, const TQString &room);
	void slotConfMessage(const TQString &who, const TQString &room, const TQString &msg);
	void slotConfLeave( YahooConferenceChatSession *s );
	void slotInviteConference( const TQString &room, const TQStringList &who, const TQStringList &members, const TQString &msg );
	void slotAddInviteConference( const TQString &room, const TQStringList &who, const TQStringList &members, const TQString &msg );
	void slotGotFile(const TQString &who, const TQString &url, long expires, const TQString &msg, const TQString &fname, unsigned long fesize, const TQPixmap &);
	void slotContactAdded(const TQString &myid, const TQString &who, const TQString &msg);
	void slotRejected(const TQString &, const TQString &);
	void slotTypingNotify(const TQString &, int );
	void slotGameNotify(const TQString &, int);
	void slotMailNotify(const TQString &, const TQString &, int);
	void slotSystemMessage(const TQString &);
	void slotRemoveHandler(int fd);
	//void slotHostConnect(const TQString &host, int port);
	void slotGotWebcamInvite(const TQString &);
	void slotWebcamNotAvailable( const TQString &who );
	void slotGotWebcamImage(const TQString&, const TQPixmap&);
	void slotWebcamReadyForTransmission();
	void slotWebcamStopTransmission();
	void slotOutgoingWebcamClosing();
	void slotWebcamClosed(const TQString&, int);
	void slotWebcamPaused(const TQString&);
	void slotWebcamViewerJoined( const TQString & );
	void slotWebcamViewerLeft( const TQString & );
	void slotWebcamViewerRequest( const TQString & );
	void slotPictureStatusNotify( const TQString&, int);
	void slotGotBuddyIcon(const TQString&, const TQByteArray&, int);
	void slotGotBuddyIconInfo(const TQString&, KURL, int);
	void slotGotBuddyIconChecksum(const TQString&, int);
	void slotGotBuddyIconRequest(const TQString &);
	void slotBuddyIconChanged(const TQString&, int);
	void slotGotYABEntry( YABEntry *entry );
	void slotGotYABRevision( long revision, bool merged );
	void slotSaveYABEntry( YABEntry &entry );
	void slotModifyYABEntryError( YABEntry *entry, const TQString & );
	void slotChatJoined( int roomId, int categoryId, const TQString &comment, const TQString &handle );
	void slotChatBuddyHasJoined( const TQString &nick, const TQString &handle, bool suppressNotification );
	void slotChatBuddyHasLeft( const TQString &nick, const TQString &handle );
	void slotChatMessageReceived( const TQString &nick, const TQString &message, const TQString &handle );
	void slotLeavChat();

	void slotReceiveFileAccepted( Kopete::Transfer *trans, const TQString& fileName );
	void slotReceiveFileRefused( const Kopete::FileTransferInfo& info );
	void slotFileTransferComplete( unsigned int id );
	void slotFileTransferError( unsigned int id, int error, const TQString &desc );
	void slotFileTransferBytesProcessed( unsigned int id, unsigned int bytes );
	void slotFileTransferResult( TDEIO::Job * );
	void slotError( int level );

private:

	/**
	 * Handle the signal and slot connections and disconnects
	 */
	void initConnectionSignals( enum SignalConnectionType sct );

	void setupActions( bool connected );

	TQString prepareIncomingMessage( const TQString &msg );

	/**
	 * internal (to the plugin) controls/flags
	 * This should be kept in sync with server - if a buddy is removed, this should be changed accordingly.
	 */
	TQMap<TQString, TQPair<TQString, TQString> > IDs;

	/**
	 * Conferences list, maped by room name (id)
	 */
	TQMap<TQString, YahooConferenceChatSession *> m_conferences;
	YahooChatChatSession * m_chatChatSession;
	TQStringList m_pendingConfInvites;
	TQStringList m_pendingWebcamInvites;
	TQStringList m_pendingFileTransfers;

	TQMap<unsigned int, Kopete::Transfer *> m_fileTransfers;

	bool theHaveContactList;	// Do we have the full server-side contact list yet?
	int stateOnConnection;		// The state to change to on connection

	/**
	 * External Settings and Descriptors
	 */
	bool m_useServerGroups;		// Use the groups on the server for import
	bool m_importContacts;		// Import the contacts from the server
	int m_sessionId;		// The Yahoo session descriptor
	int m_lastDisconnectCode;	// The last disconnect code.
	int m_currentMailCount;
	long m_YABLastMerge;		// The YAB Revision on which the last merge was done
	long m_YABLastRemoteRevision;	// The last remote YAB Revision on which a sync was done
	YahooProtocol *m_protocol;	// The Protocol Object

	YahooWebcam *m_webcam;

	TDEAction *m_openInboxAction;	// Menu item openInbox
	TDEAction *m_openYABAction;	// Menu item openYahooAddressbook
	TDEAction *m_editOwnYABEntry;	// Menu item editOwnYABEntry
	TDEAction *m_joinChatAction;	// Menu item joinChatAction

	Client *m_session;		// The Connection object
};

#endif // YAHOOACCOUNT_H
