/* This file is part of the KDE project
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>
   Copyright (C) 2004  Alexander Dymo <cloudtemple@mskat.net>
   Copyright (C) 2006 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "booledit.h"
#include "../property.h"

#include <kiconloader.h>
#include <tdelocale.h>
#include <kcombobox.h>
#include <kdebug.h>

#include <tqtoolbutton.h>
#include <tqpainter.h>
#include <tqvariant.h>
#include <tqlayout.h>
#include <tqbitmap.h>

using namespace KoProperty;

BoolEdit::BoolEdit(Property *property, TQWidget *parent, const char *name)
 : Widget(property, parent, name)
 , m_yesIcon( SmallIcon("button_ok") )
 , m_noIcon( SmallIcon("button_no") )
{
    m_toggle = new TQToolButton(this);
    m_toggle->setToggleButton( true );
    m_toggle->setFocusPolicy(TQWidget::WheelFocus);
    m_toggle->setUsesTextLabel(true);
    m_toggle->setTextPosition(TQToolButton::Right);
    m_toggle->setSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Fixed);
    //we're not using layout to because of problems with button size
    m_toggle->move(0, 0);
    m_toggle->resize(width(), height());
    setFocusWidget(m_toggle);
    connect(m_toggle, TQ_SIGNAL(stateChanged(int)), this, TQ_SLOT(slotValueChanged(int)));
}

BoolEdit::~BoolEdit()
{
}

TQVariant
BoolEdit::value() const
{
    return TQVariant(m_toggle->isOn());
}

void
BoolEdit::setValue(const TQVariant &value, bool emitChange)
{
    m_toggle->blockSignals(true);
    m_toggle->setOn(value.toBool());
    setState( value.toBool() ? TQButton::On : TQButton::Off );
    m_toggle->blockSignals(false);
    if (emitChange)
        emit valueChanged(this);
}

void
BoolEdit::slotValueChanged(int state)
{
    setState(state);
    emit valueChanged(this);
}

static void drawViewerInternal(TQPainter *p, const TQRect &r, const TQVariant &value,
 const TQPixmap& yesIcon, const TQPixmap& noIcon, const TQString& nullText)
{
    p->eraseRect(r);
    TQRect r2(r);
    r2.moveLeft(TDEIcon::SizeSmall + 6);

    if(value.isNull() && !nullText.isEmpty()) {
        p->drawText(r2, TQt::AlignVCenter | TQt::AlignLeft, nullText);
    }
    else if(value.toBool()) {
        p->drawPixmap(3, (r.height()-1-TDEIcon::SizeSmall)/2, yesIcon);
        p->drawText(r2, TQt::AlignVCenter | TQt::AlignLeft, i18n("Yes"));
    }
    else {
        p->drawPixmap(3, (r.height()-1-TDEIcon::SizeSmall)/2, noIcon);
        p->drawText(r2, TQt::AlignVCenter | TQt::AlignLeft, i18n("No"));
    }
}

void
BoolEdit::drawViewer(TQPainter *p, const TQColorGroup &cg, const TQRect &r, const TQVariant &value)
{
	Q_UNUSED(cg);
	drawViewerInternal(p, r, value, m_yesIcon, m_noIcon, "");
}

void
BoolEdit::setState(int state)
{
    if(TQButton::On == state) {
        m_toggle->setIconSet(TQIconSet(m_yesIcon));
        m_toggle->setTextLabel(i18n("Yes"));
    }
    else if (TQButton::Off == state) {
        m_toggle->setIconSet(TQIconSet(m_noIcon));
        m_toggle->setTextLabel(i18n("No"));
    }
}

void
BoolEdit::resizeEvent(TQResizeEvent *ev)
{
    m_toggle->resize(ev->size());
}

bool
BoolEdit::eventFilter(TQObject* watched, TQEvent* e)
{
    if(e->type() == TQEvent::KeyPress) {
        TQKeyEvent* ev = static_cast<TQKeyEvent*>(e);
        const int k = ev->key();
        if(k == TQt::Key_Space || k == TQt::Key_Enter || k == TQt::Key_Return) {
            if (m_toggle)
                m_toggle->toggle();
            return true;
        }
    }
    return Widget::eventFilter(watched, e);
}

void
BoolEdit::setReadOnlyInternal(bool readOnly)
{
    setVisibleFlag(!readOnly);
}

//--------------------------------------------------

ThreeStateBoolEdit::ThreeStateBoolEdit(Property *property, TQWidget *parent, const char *name)
 : ComboBox(property, parent, name)
 , m_yesIcon( SmallIcon("button_ok") )
 , m_noIcon( SmallIcon("button_no") )
{
	m_edit->insertItem( m_yesIcon, i18n("Yes") );
	m_edit->insertItem( m_noIcon, i18n("No") );
	TQVariant thirdState = property ? property->option("3rdState") : TQVariant();
	TQPixmap nullIcon( m_yesIcon.size() ); //transparent pixmap of appropriate size
	nullIcon.setMask(TQBitmap(m_yesIcon.size(), true));
	m_edit->insertItem( nullIcon, thirdState.toString().isEmpty() ? i18n("None") : thirdState.toString() );
}

ThreeStateBoolEdit::~ThreeStateBoolEdit()
{
}

TQVariant
ThreeStateBoolEdit::value() const
{
	// list items: true, false, NULL
	const int idx = m_edit->currentItem();
	if (idx==0)
		return TQVariant(true);
	else
		return idx==1 ? TQVariant(false) : TQVariant();
}

void
ThreeStateBoolEdit::setProperty(Property *prop)
{
	m_setValueEnabled = false; //setValue() couldn't be called before fillBox()
	Widget::setProperty(prop);
	m_setValueEnabled = true;
	if(prop)
		setValue(prop->value(), false); //now the value can be set
}

void
ThreeStateBoolEdit::setValue(const TQVariant &value, bool emitChange)
{
	if (!m_setValueEnabled)
		return;

	if (value.isNull())
		m_edit->setCurrentItem(2);
	else
		m_edit->setCurrentItem(value.toBool() ? 0 : 1);

	if (emitChange)
		emit valueChanged(this);
}

void
ThreeStateBoolEdit::drawViewer(TQPainter *p, const TQColorGroup &cg, const TQRect &r, const TQVariant &value)
{
	Q_UNUSED(cg);
	drawViewerInternal(p, r, value, m_yesIcon, m_noIcon, m_edit->text(2));
}

#include "booledit.moc"
