/* ============================================================
 * File  : cameraui.cpp
 * Author: Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Date  : 2003-01-21
 * Description :
 *
 * Copyright 2003 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright 2004 by Tudor Calin <tudor@1xtech.com>
 *
 * Update : 08/28/2003 - Gilles Caulier <caulier.gilles@free.fr>
 *          Add standard shortcuts and toolbars KDE menu entry.
 *          Add new toolbar icons.
 *          Improve i18n messages.
 *          17/09/2003 - Gilles Caulier <caulier.gilles@free.fr>
 *          Add FullScreen mode.
 *          19/09/2003 - Gilles Caulier <caulier.gilles@free.fr>
 *          Add new default shortcuts.
 *
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqcombobox.h>
#include <tqdir.h>
#include <tqfileinfo.h>
#include <tqimage.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqsplitter.h>
#include <tqstringlist.h>
#include <tqtooltip.h>
#include <tqvaluelist.h>
#include <tqvbox.h>
#include <tqpushbutton.h>
#include <tqframe.h>

// KDE includes

#include <tdeaccel.h>
#include <kcombobox.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <kdirlister.h>
#include <tdefiledialog.h>
#include <tdefileitem.h>
#include <tdelocale.h>
#include <kedittoolbar.h>
#include <tdeio/job.h>
#include <kkeydialog.h>
#include <klineeditdlg.h>
#include <tdemessagebox.h>
#include <tdepopupmenu.h>
#include <kprogress.h>
#include <krun.h>
#include <kservice.h>
#include <tdestandarddirs.h>
#include <kstatusbar.h>
#include <tdestdaccel.h>
#include <kurl.h>
#include <tdeapplication.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <kguiitem.h>

// Local includes.

#include "kpaboutdata.h"
#include "pluginsversion.h"
#include "camerafolderitem.h"
#include "cameraiconview.h"
#include "camerafolderview.h"
#include "cameraiconitem.h"
#include "cameralist.h"
#include "cameratype.h"
#include "dmessagebox.h"
#include "gpcontroller.h"
#include "gpeventfilter.h"
#include "gpfileitemcontainer.h"
#include "gpfileiteminfo.h"
#include "savefiledialog.h"
#include "setupcamera.h"
#include "cameraui.h"
#include "cameraui.moc"

namespace KIPIKameraKlientPlugin
{

CameraUI::CameraUI() : TQWidget()
{
    setWFlags(TQt::WDestructiveClose);
    resize(700, 440);
    setMinimumSize(600, 400);

    mCameraList = new CameraList(this, locateLocal("data", "kipi/cameras.xml"));
    mCameraType =  new CameraType();

    //---------------------------------------------

    TQVBoxLayout *dvlay = new TQVBoxLayout( this, 6 );
    TQHBoxLayout* mMainBoxLayout = new TQHBoxLayout(dvlay);
    mMainBoxLayout->setResizeMode(TQLayout::FreeResize);
    TQVBoxLayout* mLeftBoxLayout = new TQVBoxLayout(mMainBoxLayout, 0);
    TQVBoxLayout* mBtnBoxLayout = new TQVBoxLayout(mMainBoxLayout, 4);
    mBtnBoxLayout->setMargin(2);

    // create Button Box ----------------------------------------------------------------------

    mCameraSetupBtn = new TQPushButton(i18n("Setup"), this);
    mCameraSetupBtn->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mCameraSetupBtn);
    mCameraStopBtn = new TQPushButton(i18n("Stop"), this);
    mCameraStopBtn->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mCameraStopBtn);
    mCameraDownloadBtn = new TQPushButton(i18n("Download"), this);
    mCameraDownloadBtn->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mCameraDownloadBtn);
    mCameraUploadBtn = new TQPushButton(i18n("Upload"), this);
    mCameraUploadBtn->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mCameraUploadBtn);
    mCameraDeleteBtn = new TQPushButton(i18n("Delete"), this);
    mCameraDeleteBtn->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mCameraDeleteBtn);
    TQSpacerItem* mBtnSpacer = new TQSpacerItem(0, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding);
    mBtnBoxLayout->addItem(mBtnSpacer);
    mDialogCloseBtn = new TQPushButton(i18n("Close"), this);
    mDialogCloseBtn->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mDialogCloseBtn);

    // About data and help button ---------------------------------------------------

    mhelpButton = new TQPushButton(i18n("&Help"), this);
    mhelpButton->setMinimumSize(TQSize(100, 0));
    mBtnBoxLayout->addWidget(mhelpButton);

    m_about = new KIPIPlugins::KPAboutData(I18N_NOOP("KameraKlient"),
                                           0,
                                           TDEAboutData::License_GPL,
                                           I18N_NOOP("A Digital camera interface Kipi plugin"),
                                           "(c) 2003-2004, Renchi Raju\n"
                                           "(c) 2004, Tudor Calin");

    m_about->addAuthor("Renchi Raju", I18N_NOOP("Original author from Digikam project"),
                       "renchi@pooh.tam.uiuc.edu");

    m_about->addAuthor("Tudor Calin", I18N_NOOP("Porting the Digikam GPhoto2 interface to Kipi. Maintainer"),
                       "tudor@1xtech.com");

    KHelpMenu* helpMenu = new KHelpMenu(this, m_about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQ_SLOT(slotHelp()), 0, -1, 0);
    mhelpButton->setPopup( helpMenu->menu() );

    // create Camera Box-----------------------------------------------------------------------

    TQHBoxLayout* mCameraBoxLayout = new TQHBoxLayout(mLeftBoxLayout, 4);
    mCameraBoxLayout->setMargin(4);
    mCameraConnectBtn = new TQPushButton(i18n("Connect"), this);
    mCameraBoxLayout->addWidget(mCameraConnectBtn);
    mCameraComboBox = new TQComboBox(this, "camera");
    mCameraComboBox->setInsertionPolicy(TQComboBox::AtBottom);
    mCameraComboBox->setSizePolicy(TQSizePolicy(TQSizePolicy::Expanding, TQSizePolicy::Fixed));
    mCameraBoxLayout->addWidget(mCameraComboBox);

    // create Download Directory Camera Box ---------------------------------------------------

    TQHBoxLayout *mDownloadDirectoryBoxLayout = new TQHBoxLayout(mLeftBoxLayout, 4);
    mDownloadDirectoryBoxLayout->setMargin(4);
    TQLabel* mDownloadDirectoryLabel = new TQLabel(i18n("Download to: "), this);
    mDownloadDirectoryBoxLayout->addWidget(mDownloadDirectoryLabel);
    mDownloadDirectoryEdit = new TQLineEdit(this);
    mDownloadDirectoryEdit->setSizePolicy(TQSizePolicy(TQSizePolicy::Expanding, TQSizePolicy::Fixed));
    mDownloadDirectoryEdit->setReadOnly(true);
    mDownloadDirectoryBoxLayout->addWidget(mDownloadDirectoryEdit);
    mChangeDownloadDirectoryBtn = new TQPushButton(i18n("&Change"), this);
    mDownloadDirectoryBoxLayout->addWidget(mChangeDownloadDirectoryBtn); // -------------------------
    mSplitter = new TQSplitter(this);
    mLeftBoxLayout->addWidget(mSplitter);
    mSplitter->setSizePolicy(TQSizePolicy(TQSizePolicy::Expanding, TQSizePolicy::Expanding));
    mFolderView   = new CameraFolderView(mSplitter);
    mIconView   = new CameraIconView(mSplitter);
    mSplitter->setOpaqueResize(true);
    mSplitter->setResizeMode(mFolderView, TQSplitter::Stretch);
    mSplitter->setResizeMode(mIconView, TQSplitter::Stretch);
    container_  = new GPFileItemContainer(this, mFolderView, mIconView);
    efilter_    = new GPEventFilter(this);
    controller_ = new GPController(this, *mCameraType);
    controller_->start();
    cameraConnected_ = false;

    // create Status Bar -----------------------------------------------------------------------------

    mStatusBar =  new KStatusBar(this);
    mLeftBoxLayout->addWidget(mStatusBar);
    mStatusLabel = new TQLabel(mStatusBar);
    mStatusLabel->setText(i18n("Ready"));
    mStatusBar->addWidget(mStatusLabel, 7, true);
    mProgressBar = new KProgress(mStatusBar);
    mProgressBar->setTotalSteps(100);
    mStatusBar->addWidget(mProgressBar, 5, true);

    // -------------------------------------------------

    setupAccel();
    setupConnections();
    mCameraList->load();
    readSettings();
}

CameraUI::~CameraUI() {
    writeSettings();
    delete controller_;
    delete container_;
    mFolderView->clear();
    mIconView->clear();

    delete m_about;
}


void CameraUI::slotHelp()
{
    tdeApp->invokeHelp("kameraklient",
                                             "kipi-plugins");
}

const CameraType* CameraUI::cameraType() {
    return mCameraType;
}

void CameraUI::setupAccel() {
    mCameraUIAccel = new TDEAccel(this);
    mCameraUIAccel->insert("Select All", i18n("Select All"),
	    		i18n("Select all the images from the camera."),
			CTRL+Key_A, this, TQ_SLOT(slotSelectAll()));
    mCameraUIAccel->insert("Select None", i18n("Select None"),
	    		i18n("Deselect all the images from the camera."),
			CTRL+Key_U, this, TQ_SLOT(slotSelectNone()));
    mCameraUIAccel->insert("Invert Selection", i18n("Invert Selection"),
	    		i18n("Invert the selection."),
			CTRL+Key_Asterisk, this, TQ_SLOT(slotSelectInvert()));
    mCameraUIAccel->insert("Select New", i18n("Select New Items"),
	    		i18n("Select all the that were not previously downloaded."),
			CTRL+Key_Slash, this, TQ_SLOT(slotSelectNew()));
    setCameraConnected(false);
}

void CameraUI::setupConnections() {
    connect(this, TQ_SIGNAL(signalStatusMsg(const TQString&)), this, TQ_SLOT(slotSetStatusMsg(const TQString&)));
    connect(this, TQ_SIGNAL(signalProgressVal(int)), this, TQ_SLOT(slotSetProgressVal(int)));
    connect(this, TQ_SIGNAL(signalBusy(bool)), this, TQ_SLOT(slotBusy(bool)));
    connect(efilter_, TQ_SIGNAL(signalStatusMsg(const TQString&)), this, TQ_SIGNAL(signalStatusMsg(const TQString&)));
    connect(efilter_, TQ_SIGNAL(signalProgressVal(int)), this, TQ_SIGNAL(signalProgressVal(int)));
    connect(efilter_, TQ_SIGNAL(signalBusy(bool)), this, TQ_SIGNAL(signalBusy(bool)));
    connect(mFolderView, TQ_SIGNAL(signalFolderChanged(CameraFolderItem*)), this, TQ_SLOT(slotFolderSelected(CameraFolderItem*)));
    connect(mIconView, TQ_SIGNAL(signalDownloadSelectedItems()), this, TQ_SLOT(slotCameraDownloadSelected()));
    connect(mIconView, TQ_SIGNAL(signalDeleteSelectedItems()), this, TQ_SLOT(slotCameraDeleteSelected()));
    connect(mChangeDownloadDirectoryBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotChangeDownloadDirectory()));
    connect(mCameraList, TQ_SIGNAL(signalCameraListChanged()), this, TQ_SLOT(slotSyncCameraComboBox()));
    connect(mCameraConnectBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotCameraConnectToggle()));
    connect(mCameraSetupBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotSetupCamera()));
    connect(mCameraStopBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotCameraCancel()));
    connect(mCameraDownloadBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotCameraDownloadSelected()));
    connect(mCameraUploadBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotCameraUpload()));
    connect(mCameraDeleteBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(slotCameraDeleteSelected()));
    connect(mDialogCloseBtn, TQ_SIGNAL(pressed()), this, TQ_SLOT(close()));
}

void CameraUI::setCameraConnected(bool val) {
    mCameraDownloadBtn->setEnabled(val);
    mCameraUploadBtn->setEnabled(val);
    mCameraDeleteBtn->setEnabled(val);
    if(val) {
	mStatusLabel->setText(i18n("Connected"));
	mCameraConnectBtn->setText(i18n("Disconnect"));
    } else {
	mStatusLabel->setText(i18n("Disconnected"));
	mCameraConnectBtn->setText(i18n("Connect"));
    }
}

void CameraUI::slotSetStatusMsg(const TQString& msg) {
    mStatusLabel->setText(msg);
}

void CameraUI::slotSetProgressVal(int val) {
    if(val >= 0 && val <= 100) {
	mProgressBar->setProgress(val);
    }
}

void CameraUI::slotResetStatusBar() {
    mProgressBar->setTotalSteps(100);
    mProgressBar->setProgress(0);
}

void CameraUI::slotBusy(bool val) {
    if(!val) {
	slotResetStatusBar();
    }
    mCameraStopBtn->setEnabled(val);
}

void CameraUI::slotSetupCamera() {
    SetupCamera *mSetupDialog = new SetupCamera(this, "camerasetup");
    mSetupDialog->exec();
}

void CameraUI::slotSyncCameraComboBox() {
    mCameraComboBox->clear();
    TQPtrList<CameraType>* mCameraTypeList = mCameraList->cameraList();
    for(mCameraTypeList->first(); mCameraTypeList->current(); mCameraTypeList->next()) {
	mCameraComboBox->insertItem(mCameraTypeList->current()->model());
    }
}

void CameraUI::setCameraType(const CameraType& ctype) {
    *mCameraType = ctype;
    delete controller_;
    controller_ = new GPController(this, *mCameraType);
    controller_->start();
}

void CameraUI::cameraInitialized(bool val) {
    if(val) {
        cameraConnected_ = true;
        setCameraConnected(true);
        container_->addVirtualFolder(mCameraType->model());
        container_->addRootFolder("/");
        controller_->requestGetSubFolders("/");
        controller_->requestGetAllItemsInfo("/");
        mFolderView->virtualFolder()->setSelected(true);
    }
}

void CameraUI::cameraSubFolder(const TQString& folder, const TQString& subFolder) {
    container_->addFolder(folder, subFolder);
}

void CameraUI::cameraNewItems(const TQString& folder, const GPFileItemInfoList& infoList) {
    TQListViewItem *item = mFolderView->currentItem();
    if (!item) {
	return;
    }
    CameraFolderItem *folderItem = static_cast<CameraFolderItem *>(item);
    if (folderItem->folderPath() != folder && !folderItem->isVirtualFolder()) {
        return;
    }
    container_->addFiles(folder, infoList);
    GPFileItemInfoList::const_iterator it;
    for (it = infoList.begin(); it != infoList.end(); ++it) {
        if ((*it).mime.contains("image")) {
            controller_->requestGetThumbnail(folder, (*it).name);
	}
    }
}

void CameraUI::cameraNewItems(const GPFileItemInfoList& infoList) {
    TQListViewItem *item = mFolderView->currentItem();
    if (!item) {
	return;
    }
    CameraFolderItem *folderItem = static_cast<CameraFolderItem *>(item);
    if (!folderItem->isVirtualFolder()){
        return;
    }
    container_->addFiles(infoList);
    GPFileItemInfoList::const_iterator it;
    for (it = infoList.begin(); it != infoList.end(); ++it) {
        if ((*it).mime.contains("image")) {
            controller_->requestGetThumbnail((*it).folder, (*it).name);
	}
    }
}

void CameraUI::cameraNewThumbnail(const TQString& folder, const TQString& itemName, const TQImage& thumbnail) {
    CameraIconItem *iconItem = container_->findItem(folder, itemName);
    if (!iconItem) {
	return;
    }
    mIconView->setThumbnail(iconItem, thumbnail);
}

void CameraUI::cameraDownloadedItem(const TQString& folder, const TQString& itemName) {
    CameraIconItem *iconItem = container_->findItem(folder, itemName);
    if(!iconItem) {
	return;
    }
    mIconView->markDownloaded(iconItem);
}

void CameraUI::cameraDeletedItem(const TQString& folder, const TQString& itemName) {
    container_->delFile(folder, itemName);
}

void CameraUI::cameraErrorMsg(const TQString& msg) {
    DMessageBox::showMsg(msg);
}

void CameraUI::slotCameraConnectToggle() {
    if (mCameraComboBox->count() == 0) {
	KMessageBox::error(this, i18n("There is no configured camera!"));
	return;
    }
    mCameraType = mCameraList->find(mCameraComboBox->currentText());
    setCameraType(*mCameraType);
    setCameraConnected(false);
    if(!cameraConnected_) {
        controller_->requestInitialize();
    } else {
        delete controller_;
        controller_ = new GPController(this, *mCameraType);
        controller_->start();
        cameraConnected_ = false;
        mIconView->clear();
        mFolderView->clear();
    }
}

void CameraUI::slotCameraDownloadSelected() {
    if(!cameraConnected_) {
	return;
    }
    TQString dir = mDownloadDirectoryEdit->text();
    TQDir qdir(dir);
    if(!qdir.exists()) {
        KMessageBox::error(this, i18n("'%1' directory does not exist.").arg(dir));
        return;
    }
    int count = 0;
    for(ThumbItem *i = mIconView->firstItem(); i; i=i->nextItem() ) {
        if (i->isSelected()) {
	    ++count;
	}
    }
    if(count == 0) {
	return;
    }
    bool proceed = true;
    bool overwriteAll = false;
    for(ThumbItem *i = mIconView->firstItem(); i; i=i->nextItem()) {
        if(i->isSelected()) {
            CameraIconItem *item = static_cast<CameraIconItem*>(i);
            if(!item) {
		continue;
	    }
            downloadOneItem(item->fileInfo()->name, item->fileInfo()->folder, dir, proceed, overwriteAll);
            if(!proceed) {
		return;
	    }
        }
    }
}

void CameraUI::slotCameraDeleteSelected() {
    if(!cameraConnected_) {
       return;
    }
    TQStringList deleteList;
    for (ThumbItem *i = mIconView->firstItem(); i;
	i=i->nextItem()) {
	if(i->isSelected()) {
	    CameraIconItem *item = static_cast<CameraIconItem*>(i);
	    deleteList.append(item->fileInfo()->name);
        }
    }
    if (deleteList.isEmpty()) {
	return;
    }
    TQString warnMsg(i18n("About to delete these Image(s)\n" "Are you sure?"));
    if(KMessageBox::warningContinueCancelList(this, warnMsg, deleteList, i18n("Warning"), KGuiItem(i18n("Delete"),"edit-delete")) ==  KMessageBox::Continue) {
	CameraIconItem *item = static_cast<CameraIconItem*>(mIconView->firstItem());
        while(item) {
            CameraIconItem *nextItem = static_cast<CameraIconItem *>(item->nextItem());
            if (item->isSelected()) {
                controller_->requestDeleteItem(item->fileInfo()->folder, item->fileInfo()->name);
	    }
            item = nextItem;
        }
    }
}

void CameraUI::slotCameraUpload() {
    TQString reason;
    if (! cameraReadyForUpload(reason) ) {
        KMessageBox::error(0, reason);
        return;
    }
    CameraFolderItem *folderItem = static_cast<CameraFolderItem *>(mFolderView->selectedItem());
    TQStringList list = KFileDialog::getOpenFileNames(TQString());
    bool ok;
    for (TQStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
        TQFileInfo info(*it);
        if(!info.exists()) {
	    continue;
	}
        if(info.isDir()) {
	    continue;
	}
        TQString uploadName = info.fileName();
        while (container_->findItem(folderItem->folderPath(), uploadName)) {
            TQString msg(i18n("Camera Folder '%1' contains item '%2'\n Please, enter New Name").arg(folderItem->folderName()).arg(uploadName));
            uploadName = KLineEditDlg::getText(msg,uploadName,&ok,this);
            if(!ok){
		return;
	    }
        }
        controller_->requestUploadItem(folderItem->folderPath(), info.absFilePath(), uploadName);
    }
}

void CameraUI::slotCameraCancel() {
    controller_->cancel();
}

void CameraUI::slotSelectAll() {
    mIconView->selectAll();
}

void CameraUI::slotSelectNone() {
    mIconView->clearSelection();
}

void CameraUI::slotSelectInvert() {
    mIconView->invertSelection();
}

void CameraUI::slotSelectNew() {
    mIconView->clearSelection();
    for (ThumbItem *it = mIconView->firstItem(); it; it = it->nextItem()) {
        CameraIconItem *item = static_cast<CameraIconItem *>(it);
        if (item->fileInfo()->downloaded == 0) {
            item->setSelected(true, false);
	}
    }
}

void CameraUI::slotFolderSelected(CameraFolderItem *folderItem) {
    if (!folderItem) {
	return;
    }
    controller_->cancel();
    mIconView->clear();
    if (folderItem->isVirtualFolder()) {
        controller_->requestGetAllItemsInfo("/");
    } else {
        controller_->requestGetItemsInfo(folderItem->folderPath());
    }
}

void CameraUI::downloadOneItem(const TQString& item, const TQString& folder, const TQString& downloadDir, bool& proceedFurther, bool& overwriteAll) {
    proceedFurther = true;
    TQString saveFile(downloadDir);
    if (!downloadDir.endsWith("/")) {
        saveFile += "/";
    }
    saveFile += item;
    while (TQFile::exists(saveFile) && !(overwriteAll)) {
        bool overwrite=false;
        SavefileDialog *dlg = new SavefileDialog(saveFile);
        if (dlg->exec()== TQDialog::Rejected) {
            delete dlg;
            proceedFurther = false;
            return;
        }
        switch(dlg->saveFileOperation()) {
	    case (SavefileDialog::Skip): {
		delete dlg;
		return;
	    }
	    case (SavefileDialog::SkipAll): {
		delete dlg;
		proceedFurther = false;
		return;
	    }
	    case (SavefileDialog::Overwrite): {
		overwrite = true;
		delete dlg;
		break;
	    }
	    case (SavefileDialog::OverwriteAll): {
		overwriteAll = true;
		delete dlg;
		break;
	    }
	    case (SavefileDialog::Rename): {
		saveFile = downloadDir+"/"+dlg->renameFile();
		delete dlg;
		break;
	    }
	    default:  {
		delete dlg;
		proceedFurther = false;
		return;
	    }
        }
        if (overwrite) {
	    break;
	}
    }
    controller_->requestDownloadItem(folder, item, saveFile);
}

bool CameraUI::cameraReadyForUpload(TQString& reason) {
    bool result = false;
    if (!cameraConnected_) {
	reason = i18n("Camera Not Initialized");
	return result;
    } /*
    if (!controller_->cameraSupportsUpload()) {
	reason = i18n("Camera does not support Uploads");
	return result;
    } */
    if (!mFolderView->selectedItem() || mFolderView->selectedItem() == mFolderView->firstChild()) {
	reason = i18n("Please Select a Folder on Camera to Upload");
	return result;
    }
    result = true;
    return result;
}

void CameraUI::slotChangeDownloadDirectory() {
    TQString result = KFileDialog::getExistingDirectory(mDownloadDirectoryEdit->text(), this);
    if(!((new TQFileInfo(result))->isWritable())) {
	KMessageBox::sorry(this, i18n("Sorry! The directory is not writable!"));
	return;
    }
    if(!result.isEmpty()) {
	mDownloadDirectoryEdit->setText(result);
    }
}

void CameraUI::writeSettings() {
    mConfig = new TDEConfig("kipirc");
    mConfig->setGroup("KameraKlient Settings");
    mConfig->writePathEntry("DownloadDirectory", mDownloadDirectoryEdit->text());
    mConfig->writeEntry("DialogSize", frameSize());
    mConfig->writeEntry("DialogXPos", x());
    mConfig->writeEntry("DialogYPos", y());
    mConfig->writeEntry("SplitterSizes", mSplitter->sizes());
    mConfig->sync();
    delete mConfig;
}

void CameraUI::readSettings() {
    mConfig = new TDEConfig("kipirc");
    mConfig->setGroup("KameraKlient Settings");
    mDownloadDirectoryEdit->setText(mConfig->readPathEntry("DownloadDirectory", "$HOME"));
    resize(mConfig->readSizeEntry("DialogSize"));
    move(mConfig->readNumEntry("DialogXPos"), mConfig->readNumEntry("DialogYPos"));
    mSplitter->setSizes(mConfig->readIntListEntry("SplitterSizes"));
    delete mConfig;
}

}  // NameSpace KIPIKameraKlientPlugin

