/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <tqfile.h>
#include <tqapplication.h>
#include <tqspinbox.h>
#include <tqregexp.h>
#include <tqcheckbox.h>
#include <tqlabel.h>
#include <tqlayout.h>

#include <tdelocale.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kiconloader.h>
#include <kcombobox.h>
#include <tdemessagebox.h>
#include <tdeio/netaccess.h>
#include <kurl.h>

#include "settings.h"
#include "fileimporterbibtex.h"
#include "webquerydblp.h"

namespace KBibTeX
{
    WebQueryDBLPWidget::WebQueryDBLPWidget( TQWidget *parent, const char *name )
            : WebQueryWidget( parent, name )
    {
        init();

        Settings *settings = Settings::self();
        TQString value = settings->getWebQueryDefault( "DBLP_query" );
        value = value == TQString::null ? "" : value;
        lineEditQuery->setText( value );
        slotTextChanged( value, true );
        value = settings->getWebQueryDefault( "DBLP_keepSeparate" );
        value = value == TQString::null || value.isEmpty() ? "0" : value;
        checkBoxKeepEntriesSeparate->setChecked( value == "1" );
    }

    void WebQueryDBLPWidget::init()
    {
        TQVBoxLayout *vLayout = new TQVBoxLayout( this, 0, KDialog::spacingHint() );

        TQHBoxLayout *hLayout = new TQHBoxLayout( );
        vLayout->addLayout( hLayout );

        KPushButton *clearSearchText = new KPushButton( this );
        clearSearchText->setIconSet( TQIconSet( SmallIcon( "locationbar_erase" ) ) );
        hLayout->addWidget( clearSearchText );
        TQLabel *label = new TQLabel( i18n( "Search &term:" ), this );
        hLayout->addWidget( label );
        lineEditQuery = new KLineEdit( this );
        hLayout->addWidget( lineEditQuery );
        label->setBuddy( lineEditQuery );
        hLayout->addSpacing( KDialog::spacingHint() * 2 );
        connect( clearSearchText, TQ_SIGNAL( clicked() ), lineEditQuery, TQ_SLOT( clear() ) );
        connect( lineEditQuery, TQ_SIGNAL( textChanged( const TQString& ) ), this, TQ_SLOT( slotTextChanged( const TQString& ) ) );
        hLayout->setStretchFactor( lineEditQuery, 4 );
        TDECompletion *completionQuery = lineEditQuery->completionObject();

        label = new TQLabel( i18n( "&Number of results:" ), this );
        hLayout->addWidget( label );
        spinBoxMaxHits = new TQSpinBox( 1, 250, 1, this );
        spinBoxMaxHits->setValue( 10 );
        hLayout->addWidget( spinBoxMaxHits );
        label->setBuddy( spinBoxMaxHits );

        hLayout = new TQHBoxLayout( );
        vLayout->addLayout( hLayout );

        checkBoxKeepEntriesSeparate = new TQCheckBox( i18n( "Do not merge corresponding entries" ), this );
        hLayout->addWidget( checkBoxKeepEntriesSeparate );

        vLayout->addStretch( 0 );

        connect( lineEditQuery, TQ_SIGNAL( returnPressed() ), this, TQ_SIGNAL( startSearch() ) );
        connect( lineEditQuery, TQ_SIGNAL( returnPressed( const TQString& ) ), completionQuery, TQ_SLOT( addItem( const TQString& ) ) );
    }

    WebQueryDBLP::WebQueryDBLP( TQWidget* parent ): WebQuery( parent )
    {
        m_widget = new WebQueryDBLPWidget( parent );
    }

    WebQueryDBLP::~WebQueryDBLP()
    {
        delete m_widget;
    }

    TQString WebQueryDBLP::title()
    {
        return i18n( "DBLP" );
    }

    TQString WebQueryDBLP::disclaimer()
    {
        return i18n( "DBLP - Copyright" );
    }

    TQString WebQueryDBLP::disclaimerURL()
    {
        return "http://www.informatik.uni-trier.de/~ley/db/copyright.html";
    }

    WebQueryWidget *WebQueryDBLP::widget()
    {
        return m_widget;
    }

    void WebQueryDBLP::query()
    {
        WebQuery::query();
        Settings *settings = Settings::self();
        settings->setWebQueryDefault( "DBLP_query", m_widget->lineEditQuery->text() );
        settings->setWebQueryDefault( "DBLP_keepSeparate", m_widget->checkBoxKeepEntriesSeparate->isChecked() ? "1" : "0" );

        int numberOfResults = m_widget->spinBoxMaxHits->value();
        setNumStages( numberOfResults + 1 );

        TQString searchTerm = m_widget->lineEditQuery->text().stripWhiteSpace().replace( '$', "" );
        if ( searchTerm.isEmpty() )
        {
            setEndSearch( WebQuery::statusInvalidQuery );
            return;
        }

        KURL url = KURL( TQString( "http://dblp.l3s.de/?newsearch=1&q=%1&search_opt=all&synt_query_exp=full" ).arg( searchTerm.replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" ) ) );

        TQString completeText = download( url );
        if ( completeText != TQString::null && !m_aborted )
        {
            TQRegExp findBibTeXurl( "<a href=\"(http://dblp.uni-trier.de/rec/bibtex/[^\"]+)\"" );
            findBibTeXurl.search( completeText );
            int pos = findBibTeXurl.pos( 1 );

            int maxCount = numberOfResults;
            while ( !m_aborted && pos > -1 && ( maxCount-- ) > 0 )
            {
                KURL keyUrl = KURL( findBibTeXurl.cap( 1 ) );
                BibTeX::File *tmpBibFile = downloadBibTeXFile( keyUrl );
                if ( tmpBibFile != NULL && !m_aborted )
                {
                    if ( tmpBibFile->count() == 2 )
                    {
                        /** if the data returned from DBLP contains only two entry, check if they are InCollection and Book. In this case, one is the conference article and the other is the conference proceedings. */
                        BibTeX::Entry *firstEntry = dynamic_cast<BibTeX::Entry*>( *tmpBibFile->begin() );
                        BibTeX::Entry *secondEntry = dynamic_cast<BibTeX::Entry*>( *( ++tmpBibFile->begin() ) );

                        if ( !m_widget->checkBoxKeepEntriesSeparate->isChecked() && firstEntry != NULL && secondEntry != NULL && ( firstEntry->entryType() == BibTeX::Entry::etInProceedings || firstEntry->entryType() == BibTeX::Entry::etInCollection ) && ( secondEntry->entryType() == BibTeX::Entry::etBook || secondEntry->entryType() == BibTeX::Entry::etProceedings ) )
                        {
                            /** We merge both conference article and proceedings, as otherwise 2*n entries will be shown in the search result and it is no longer clear, which conference and with article match. */
                            BibTeX::Entry *myEntry = tmpBibFile->completeReferencedFieldsConst( firstEntry );
                            myEntry->merge( secondEntry, BibTeX::Entry::msAddNew );
                            myEntry->deleteField( BibTeX::EntryField::ftCrossRef );
                            emit foundEntry( myEntry, false );
                        }
                        else
                        {
                            emit foundEntry( new BibTeX::Entry( firstEntry ), false );
                            emit foundEntry( new BibTeX::Entry( secondEntry ), m_widget->checkBoxKeepEntriesSeparate->isChecked() );
                        }
                    }
                    else
                        for ( BibTeX::File::ElementList::iterator it = tmpBibFile->begin(); it != tmpBibFile->end(); ++it )
                        {
                            BibTeX::Entry *entry = dynamic_cast<BibTeX::Entry*>( *it );
                            if ( entry != NULL )
                                emit foundEntry( new BibTeX::Entry( entry ), false );
                        }
                }

                if ( tmpBibFile != NULL )
                    delete tmpBibFile;

                findBibTeXurl.search( completeText, pos + 1 );
                pos = findBibTeXurl.pos( 1 );
            }

            setEndSearch( WebQuery::statusSuccess );
        }
        else if ( !m_aborted )
        {
            TQString message = TDEIO::NetAccess::lastErrorString();
            if ( message.isEmpty() )
                message.prepend( '\n' );
            message.prepend( TQString( i18n( "Querying database '%1' failed." ) ).arg( title() ) );
            KMessageBox::error( m_parent, message );
            setEndSearch( WebQuery::statusError );
        }
        else
            setEndSearch( WebQuery::statusAborted );
    }

}
#include "webquerydblp.moc"
