/*
 *  Copyright (c) 2005 Boudewijn Rempt <boud@valdyas.o>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef KIS_COLORSPACE_WET_STICKY_H_
#define KIS_COLORSPACE_WET_STICKY_H_

#include <tqcolor.h>

#include "kis_global.h"
#include "kis_abstract_colorspace.h"

namespace WetAndSticky {

    /**
         * A color is specified as a vector in HLS space.  Hue is a value
     * in the range 0..360 degrees with 0 degrees being red.  Saturation
         * and Lightness are both in the range [0,1].  A lightness of 0 means
     * black, with 1 being white.  A totally saturated color has saturation
     * of 1.
     */

    enum enumDirection {
        UP,
        DOWN,
        LEFT,
        RIGHT
    };

    /**
     * Defines the contents and attributes of a cell on the canvas.
     */
    typedef struct cell {
        TQ_UINT8 blue;
        TQ_UINT8 green;
        TQ_UINT8 red;
        TQ_UINT8 alpha;

        float   hue;
        float   saturation;
        float   lightness;

        TQ_UINT8 liquid_content;
        TQ_UINT8 drying_rate;
        TQ_UINT8 miscibility;

        enumDirection direction;
        TQ_UINT8 strength;

        TQ_UINT8  absorbancy;  /* How much paint can this cell hold? */
        TQ_UINT8  volume;      /* The volume of paint. */

    } CELL, *CELL_PTR;


}



class KisWetStickyColorSpace : public KisAbstractColorSpace {
public:
    KisWetStickyColorSpace();
    virtual ~KisWetStickyColorSpace();

public:



    virtual void fromTQColor(const TQColor& c, TQ_UINT8 *dst, KisProfile *  profile = 0);
    virtual void fromTQColor(const TQColor& c, TQ_UINT8 opacity, TQ_UINT8 *dst, KisProfile *  profile = 0);

    virtual void toTQColor(const TQ_UINT8 *src, TQColor *c, KisProfile *  profile = 0);
    virtual void toTQColor(const TQ_UINT8 *src, TQColor *c, TQ_UINT8 *opacity, KisProfile *  profile = 0);

    virtual TQ_UINT8 getAlpha(const TQ_UINT8 *pixel) const;
    virtual void setAlpha(TQ_UINT8 * pixels, TQ_UINT8 alpha, TQ_INT32 nPixels) const;

    virtual void applyAlphaU8Mask(TQ_UINT8 * pixels, TQ_UINT8 * alpha, TQ_INT32 nPixels);
    virtual void applyInverseAlphaU8Mask(TQ_UINT8 * pixels, TQ_UINT8 * alpha, TQ_INT32 nPixels);

    virtual TQ_UINT8 scaleToU8(const TQ_UINT8 * srcPixel, TQ_INT32 channelPos);
    virtual TQ_UINT16 scaleToU16(const TQ_UINT8 * srcPixel, TQ_INT32 channelPos);

    virtual TQValueVector<KisChannelInfo *> channels() const;
    virtual bool hasAlpha() const;
    virtual TQ_INT32 nChannels() const;
    virtual TQ_INT32 nColorChannels() const;
    virtual TQ_INT32 nSubstanceChannels() const;
    virtual TQ_INT32 pixelSize() const;

    virtual TQString channelValueText(const TQ_UINT8 *pixel, TQ_UINT32 channelIndex) const;
    virtual TQString normalisedChannelValueText(const TQ_UINT8 *pixel, TQ_UINT32 channelIndex) const;

    virtual TQImage convertToTQImage(const TQ_UINT8 *data, TQ_INT32 width, TQ_INT32 height,
                       KisProfile *  srcProfile, KisProfile *  dstProfile,
                       TQ_INT32 renderingIntent = INTENT_PERCEPTUAL,
                       float exposure = 0.0f);


    virtual void mixColors(const TQ_UINT8 **colors, const TQ_UINT8 *weights, TQ_UINT32 nColors, TQ_UINT8 *dst) const;
    virtual void convolveColors(TQ_UINT8** colors, TQ_INT32* kernelValues, KisChannelInfo::enumChannelFlags channelFlags, TQ_UINT8 *dst, TQ_INT32 factor, TQ_INT32 offset, TQ_INT32 nColors) const;
    virtual void invertColor(TQ_UINT8 * src, TQ_INT32 nPixels);
    virtual void darken(const TQ_UINT8 * src, TQ_UINT8 * dst, TQ_INT32 shade, bool compensate, double compensation, TQ_INT32 nPixels) const;

    virtual KisCompositeOpList userVisiblecompositeOps() const;

protected:

    virtual void bitBlt(TQ_UINT8 *dst,
                TQ_INT32 dstRowSize,
                const TQ_UINT8 *src,
                TQ_INT32 srcRowStride,
                const TQ_UINT8 *srcAlphaMask,
                TQ_INT32 maskRowStride,
                TQ_UINT8 opacity,
                TQ_INT32 rows,
                TQ_INT32 cols,
                const KisCompositeOp& op);


    virtual bool convertPixelsTo(const TQ_UINT8 * src, KisProfile *  srcProfile,
                     TQ_UINT8 * dst, KisAbstractColorSpace * dstColorSpace, KisProfile *  dstProfile,
                     TQ_UINT32 numPixels,
                     TQ_INT32 renderingIntent = INTENT_PERCEPTUAL);


private:

    void compositeOver(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT8 opacity);
    void compositeClear(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT8 opacity);
    void compositeCopy(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT8 opacity);

};

#endif // KIS_COLORSPACE_WET_STICKY_H_
