/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2006-30-08
 * Description : a progress dialog for digiKam
 * 
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqlayout.h>
#include <tqwhatsthis.h>
#include <tqheader.h>
#include <tqlabel.h>
#include <tqimage.h>
#include <tqpushbutton.h>
#include <tqlistview.h>

// KDE includes.

#include <tdelocale.h>
#include <kprogress.h>
#include <tdeapplication.h>
#include <kdialogbase.h>
#include <kiconloader.h>
#include <kstandarddirs.h>

// Local includes.

#include "ddebug.h"
#include "dprogressdlg.h"
#include "dprogressdlg.moc"

namespace Digikam
{

class DProgressDlgPriv
{
public:

    DProgressDlgPriv()
    {
        progress    = 0;
        actionsList = 0;
        logo        = 0;
        title       = 0;
        label       = 0;
        allowCancel = true;
        cancelled   = false;
    }

    bool       allowCancel;
    bool       cancelled;

    TQLabel    *logo;
    TQLabel    *title;
    TQLabel    *label;

    TQListView *actionsList;

    KProgress *progress;
};

DProgressDlg::DProgressDlg(TQWidget *parent, const TQString &caption)
            : KDialogBase(parent, 0, true, caption, Cancel)
{
    d = new DProgressDlgPriv;

    TQFrame *page      = makeMainWidget();
    TQGridLayout* grid = new TQGridLayout(page, 1, 1, 0, spacingHint());
    TQVBoxLayout *vlay = new TQVBoxLayout();
    d->actionsList    = new TQListView(page);
    d->label          = new TQLabel(page);
    d->title          = new TQLabel(page);
    d->logo           = new TQLabel(page);
    d->progress       = new KProgress(page);
    vlay->addWidget(d->logo);
    vlay->addWidget(d->progress);
    vlay->addWidget(d->title);
    vlay->addStretch();

    TDEIconLoader* iconLoader = TDEApplication::kApplication()->iconLoader();
    d->logo->setPixmap(iconLoader->loadIcon("digikam", TDEIcon::NoGroup, 128, TDEIcon::DefaultState, 0, true));

    d->actionsList->addColumn("Thumb");   // no i18n here: hiden column
    d->actionsList->addColumn("Status");  // no i18n here: hiden column
    d->actionsList->setSorting(-1);
    d->actionsList->setItemMargin(1);
    d->actionsList->setSelectionMode(TQListView::NoSelection);
    d->actionsList->header()->hide();
    d->actionsList->setResizeMode(TQListView::LastColumn);

    grid->addMultiCellLayout(vlay, 0, 1, 0, 0);
    grid->addMultiCellWidget(d->label, 0, 0, 1, 1);
    grid->addMultiCellWidget(d->actionsList, 1, 1, 1, 1);
    grid->setRowStretch(1, 10);
    grid->setColStretch(1, 10);
}

DProgressDlg::~DProgressDlg()
{
    delete d;
}

void DProgressDlg::slotCancel()
{
    d->cancelled = true;

    if (d->allowCancel)
    {
        KDialogBase::slotCancel();
    }
}

void DProgressDlg::setButtonText(const TQString &text)
{
    KDialogBase::setButtonText(Cancel, text);
}

void DProgressDlg::addedAction(const TQPixmap& pix, const TQString &text)
{
    TQImage img;
    TQListViewItem *item = new TQListViewItem(d->actionsList,
                          d->actionsList->lastItem(), TQString(), text);

    if (pix.isNull())
    {
        TQString dir = TDEGlobal::dirs()->findResourceDir("digikam_imagebroken",
                                                       "image-broken.png");
        dir = dir + "/image-broken.png";
        TQPixmap pixbi(dir);
        img = pixbi.convertToImage().scale(32, 32, TQImage::ScaleMin);
    }
    else
    {
        img = pix.convertToImage().scale(32, 32, TQImage::ScaleMin);
    }

    TQPixmap pixmap(img);
    item->setPixmap(0, pixmap);
    d->actionsList->ensureItemVisible(item);
}

void DProgressDlg::reset()
{
    d->actionsList->clear();
    d->progress->setValue(0);
}

void DProgressDlg::setTotalSteps(int total)
{
    d->progress->setTotalSteps(total);
}

void DProgressDlg::setValue(int value)
{
    d->progress->setValue(value);
}

void DProgressDlg::advance(int value)
{
    d->progress->advance(value);
}

void DProgressDlg::setLabel(const TQString &text)
{
    d->label->setText(text);
}

void DProgressDlg::setTitle(const TQString &text)
{
    d->title->setText(text);
}

void DProgressDlg::showCancelButton(bool show)
{
    showButtonCancel(show);
}

void DProgressDlg::setAllowCancel(bool allowCancel)
{
    d->allowCancel = allowCancel;
    showCancelButton(allowCancel);
}

bool DProgressDlg::allowCancel() const
{
    return d->allowCancel;
}

bool DProgressDlg::wasCancelled() const
{
    return d->cancelled;
}

KProgress *DProgressDlg::progressBar() const
{
    return d->progress;
}

void DProgressDlg::setActionListVSBarVisible(bool visible)
{
    if (!visible)
        d->actionsList->setVScrollBarMode(TQScrollView::AlwaysOff);
    else
        d->actionsList->setVScrollBarMode(TQScrollView::Auto);
}

}  // NameSpace Digikam
